<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Business_One_Page_Pro
 */

if ( ! function_exists( 'business_one_page_pro_posted_on' ) ) :
/**
 * Prints HTML with meta information for the current post-date/time and author.
 */
function business_one_page_pro_posted_on( $inner = false ){	
    $post_meta_blog_page    = get_theme_mod( 'business_one_page_pro_post_meta_blog_page', array( 'date', 'author' ) );
    $post_meta_blog_section = get_theme_mod( 'business_one_page_pro_post_meta_blog_section', array( 'date', 'comment_count' ) );
    $i = 0;
    if( $inner && $post_meta_blog_page ){
        foreach( $post_meta_blog_page as $meta ){
            business_one_page_pro_meta( $meta );  
        }
    }elseif( $post_meta_blog_section ){
        foreach( $post_meta_blog_section as $meta ){
            $i++;
            business_one_page_pro_meta( $meta );
            if( $i < count($post_meta_blog_section) ){
                esc_html_e( '/', 'business-one-page-pro' );
            }  
        }
    }
}
endif;

if( ! function_exists( 'business_one_page_pro_meta' ) ) :
/**
 * List meta information
*/
function business_one_page_pro_meta( $meta ){
    
    switch( $meta ){
        
        case 'date':
        
        $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
	
    	$time_string = sprintf( $time_string, esc_attr( get_the_date( 'c' ) ), esc_html( get_the_date() ) );

        echo '<span class="posted-on"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a></span>';
        
        break;
        
        case 'author':
        
        echo '<span class="byline" itemprop="author" itemscope itemtype="https://schema.org/Person"><span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span></span>';
        
        break;
        
        case 'comment_count':
        
        $comment_count = sprintf( _nx( '%1$s Comment', '%1$s Comments', get_comments_number(), 'comments title', 'business-one-page-pro' ), number_format_i18n( get_comments_number() ) ); 
	
        echo '<span class="comments-link"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $comment_count . '</a></span>';
        
        break;
        
    }
}
endif;

if ( ! function_exists( 'business_one_page_pro_entry_footer' ) ) :
/**
 * Prints edit links
 */
function business_one_page_pro_entry_footer() {	
	edit_post_link(
		sprintf(
			/* translators: %s: Name of current post */
			esc_html__( 'Edit %s', 'business-one-page-pro' ),
			the_title( '<span class="screen-reader-text">"', '"</span>', false )
		),
		'<span class="edit-link">',
		'</span>'
	);
}
endif;

if( ! function_exists( 'business_one_page_pro_cat_tag' ) ) :
/**
 * Categories and Tags
*/
function business_one_page_pro_cat_tag(){
    
    $cat_tag = get_theme_mod( 'business_one_page_pro_cat_tag', array( 'cat', 'tag' ) );
    
    // Hide category and tag text for pages.
	if ( 'post' === get_post_type() && $cat_tag ) {
		echo '<div class="tags-block">';
            foreach( $cat_tag as $c ){
                business_one_page_pro_tax( $c );
            }            
        echo '</div>';		
	}
}
endif;

if( ! function_exists( 'business_one_page_pro_tax' ) ) :
/**
 * List Cat and Tags
*/
function business_one_page_pro_tax( $tax ){
    switch( $tax ){
        
        case 'cat':
        
        /* translators: used between list items, there is a space after the comma */
		$categories_list = get_the_category_list( esc_html__( ', ', 'business-one-page-pro' ) );		
        
        if ( $categories_list && business_one_page_pro_categorized_blog() ) {
			printf( '<span class="cat-links"><span class="fa fa-folder-open"></span>' . esc_html__( 'Categories: %1$s', 'business-one-page-pro' ) . '</span>', $categories_list ); // WPCS: XSS OK.
		}
        
        break;
        
        case 'tag':
        
        /* translators: used between list items, there is a space after the comma */
		$tags_list = get_the_tag_list( '', esc_html__( ', ', 'business-one-page-pro' ) );
		if ( $tags_list ) {
			printf( '<span class="tags-links"><span class="fa fa-tags"></span>' . esc_html__( 'Tags: %1$s', 'business-one-page-pro' ) . '</span>', $tags_list ); // WPCS: XSS OK.
		}
        
        break;
    }    
}
endif;

if( ! function_exists( 'business_one_page_pro_categories' ) ) :
/**
 * Function that list categories
*/
function business_one_page_pro_categories(){
    $categories_list = get_the_category_list( ' ' ); 
    if ( $categories_list && business_one_page_pro_categorized_blog() ) {
        echo '<span class="category">' . $categories_list . '</span>'; // WPCS: XSS OK.
    }
}
endif;

if( ! function_exists( 'business_one_page_pro_portfolio_categories' ) ) :
/**
 * Portfolio Categories lists
*/
function business_one_page_pro_portfolio_categories(){
    global $post;
    $cat_list = get_the_term_list( $post->ID, 'portfolio-category', '', esc_html__( ', ', 'business-one-page-pro' ) );
    if( $cat_list ){
        echo '<span class="category">' . $cat_list . '</span>'; // WPCS: XSS OK.
    }
}
endif;

if( ! function_exists( 'business_one_page_pro_author_details' ) ) :
/**
 * Author Bio
*/
function business_one_page_pro_author_details(){
    ?>
    <div class="author-block">
		<h2 class="author-title"><?php esc_html_e( 'About the Author', 'business-one-page-pro' ); ?></h2>
		<div class="author-holder">
			<div class="img-holder">
                <?php echo get_avatar( get_the_author_meta( 'ID' ), 130 ); ?>               
            </div>
			<div class="text-holder">
				<h3 class="author-name"><?php echo esc_html( get_the_author_meta( 'display_name' ) ); ?></h3>
				<?php echo wpautop( wp_kses_post( get_the_author_meta( 'description' ) ) ); ?>
			</div>
		</div>
	</div>
    <?php
}
endif;

/**
 * Returns true if a blog has more than 1 category.
 *
 * @return bool
 */
function business_one_page_pro_categorized_blog() {
	if ( false === ( $all_the_cool_cats = get_transient( 'business_one_page_pro_categories' ) ) ) {
		// Create an array of all the categories that are attached to posts.
		$all_the_cool_cats = get_categories( array(
			'fields'     => 'ids',
			'hide_empty' => 1,
			// We only need to know if there is more than one category.
			'number'     => 2,
		) );

		// Count the number of categories that are attached to the posts.
		$all_the_cool_cats = count( $all_the_cool_cats );

		set_transient( 'business_one_page_pro_categories', $all_the_cool_cats );
	}

	if ( $all_the_cool_cats > 1 ) {
		// This blog has more than 1 category so business_one_page_pro_categorized_blog should return true.
		return true;
	} else {
		// This blog has only 1 category so business_one_page_pro_categorized_blog should return false.
		return false;
	}
}

if( ! function_exists( 'business_one_page_pro_excerpt' ) ):  
/**
 * business_one_page_pro_excerpt can truncate a string up to a number of characters while preserving whole words and HTML tags
 *
 * @param string $text String to truncate.
 * @param integer $length Length of returned string, including ellipsis.
 * @param string $ending Ending to be appended to the trimmed string.
 * @param boolean $exact If false, $text will not be cut mid-word
 * @param boolean $considerHtml If true, HTML tags would be handled correctly
 *
 * @return string Trimmed string.
 * 
 * @link http://alanwhipple.com/2011/05/25/php-truncate-string-preserving-html-tags-words/
 */
function business_one_page_pro_excerpt($text, $length = 100, $ending = '...', $exact = false, $considerHtml = true) {
	$text = strip_shortcodes( $text );
    $text = business_one_page_pro_strip_single( 'img', $text );
    $text = business_one_page_pro_strip_single( 'a', $text );
    
    if ($considerHtml) {
		// if the plain text is shorter than the maximum length, return the whole text
		if (strlen(preg_replace('/<.*?>/', '', $text)) <= $length) {
			return $text;
		}
		// splits all html-tags to scanable lines
		preg_match_all('/(<.+?>)?([^<>]*)/s', $text, $lines, PREG_SET_ORDER);
		$total_length = strlen($ending);
		$open_tags = array();
		$truncate = '';
		foreach ($lines as $line_matchings) {
			// if there is any html-tag in this line, handle it and add it (uncounted) to the output
			if (!empty($line_matchings[1])) {
				// if it's an "empty element" with or without xhtml-conform closing slash
				if (preg_match('/^<(\s*.+?\/\s*|\s*(img|br|input|hr|area|base|basefont|col|frame|isindex|link|meta|param)(\s.+?)?)>$/is', $line_matchings[1])) {
					// do nothing
				// if tag is a closing tag
				} else if (preg_match('/^<\s*\/([^\s]+?)\s*>$/s', $line_matchings[1], $tag_matchings)) {
					// delete tag from $open_tags list
					$pos = array_search($tag_matchings[1], $open_tags);
					if ($pos !== false) {
					unset($open_tags[$pos]);
					}
				// if tag is an opening tag
				} else if (preg_match('/^<\s*([^\s>!]+).*?>$/s', $line_matchings[1], $tag_matchings)) {
					// add tag to the beginning of $open_tags list
					array_unshift($open_tags, strtolower($tag_matchings[1]));
				}
				// add html-tag to $truncate'd text
				$truncate .= $line_matchings[1];
			}
			// calculate the length of the plain text part of the line; handle entities as one character
			$content_length = strlen(preg_replace('/&[0-9a-z]{2,8};|&#[0-9]{1,7};|[0-9a-f]{1,6};/i', ' ', $line_matchings[2]));
			if ($total_length+$content_length> $length) {
				// the number of characters which are left
				$left = $length - $total_length;
				$entities_length = 0;
				// search for html entities
				if (preg_match_all('/&[0-9a-z]{2,8};|&#[0-9]{1,7};|[0-9a-f]{1,6};/i', $line_matchings[2], $entities, PREG_OFFSET_CAPTURE)) {
					// calculate the real length of all entities in the legal range
					foreach ($entities[0] as $entity) {
						if ($entity[1]+1-$entities_length <= $left) {
							$left--;
							$entities_length += strlen($entity[0]);
						} else {
							// no more characters left
							break;
						}
					}
				}
				$truncate .= substr($line_matchings[2], 0, $left+$entities_length);
				// maximum lenght is reached, so get off the loop
				break;
			} else {
				$truncate .= $line_matchings[2];
				$total_length += $content_length;
			}
			// if the maximum length is reached, get off the loop
			if($total_length>= $length) {
				break;
			}
		}
	} else {
		if (strlen($text) <= $length) {
			return $text;
		} else {
			$truncate = substr($text, 0, $length - strlen($ending));
		}
	}
	// if the words shouldn't be cut in the middle...
	if (!$exact) {
		// ...search the last occurance of a space...
		$spacepos = strrpos($truncate, ' ');
		if (isset($spacepos)) {
			// ...and cut the text in this position
			$truncate = substr($truncate, 0, $spacepos);
		}
	}
	// add the defined ending to the text
	$truncate .= $ending;
	if($considerHtml) {
		// close all unclosed html-tags
		foreach ($open_tags as $tag) {
			$truncate .= '</' . $tag . '>';
		}
	}
	return $truncate;
}
endif; // End function_exists

/**
 * Callback function for Comment List *
 * 
 * @link https://codex.wordpress.org/Function_Reference/wp_list_comments 
 */
function business_one_page_pro_theme_comment( $comment, $args, $depth ){
	if ( 'div' == $args['style'] ) {
		$tag = 'div';
		$add_below = 'comment';
	} else {
		$tag = 'li';
		$add_below = 'div-comment';
	}
?>
	<<?php echo $tag ?> <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ) ?> id="comment-<?php comment_ID() ?>">
	<?php if ( 'div' != $args['style'] ) : ?>
	<div id="div-comment-<?php comment_ID() ?>" class="comment-body" itemscope itemtype="http://schema.org/UserComments">
	<?php endif; ?>
	<div class="comment-author vcard">
	<?php if ( $args['avatar_size'] != 0 ) echo get_avatar( $comment, $args['avatar_size'] ); ?>
	<?php printf( __( '<b class="fn" itemprop="creator" itemscope itemtype="http://schema.org/Person">%s</b>', 'business-one-page-pro' ), get_comment_author_link() ); ?>
	</div>
	<?php if ( $comment->comment_approved == '0' ) : ?>
		<em class="comment-awaiting-moderation"><?php _e( 'Your comment is awaiting moderation.', 'business-one-page-pro' ); ?></em>
		<br />
	<?php endif; ?>

	<div class="comment-metadata commentmetadata">
    <a href="<?php echo htmlspecialchars( get_comment_link( $comment->comment_ID ) ); ?>">
		<time>
        <?php
			/* translators: 1: date, 2: time */
			printf( __( '%1$s - %2$s', 'business-one-page-pro' ), get_comment_date( 'M n, Y' ), get_comment_time() ); ?>
        </time>
    </a>
	</div>
    
    <div class="comment-content"><?php comment_text(); ?></div>
    
	<div class="reply">
	<?php comment_reply_link( array_merge( $args, array( 'add_below' => $add_below, 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
	</div>
	<?php if ( 'div' != $args['style'] ) : ?>
	</div>
	<?php endif; ?>
<?php
}

/**
 * Function to get Sections 
 * 
 * @return Enables Sections
 */
function business_one_page_pro_get_sections(){
    $enabled_section = array();
    $sections = get_theme_mod( 'business_one_page_pro_sort_homepage', array( 'about', 'services', 'cta1', 'portfolio', 'team', 'clients', 'blog', 'testimonial', 'cta2', 'contact' ) );
    
    /** Added to fallback if blank string is set */
    if( ! is_array( $sections ) ){
        $sections = array( 'about', 'services', 'cta1', 'portfolio', 'team', 'clients', 'blog', 'testimonial', 'cta2', 'contact' );
    }
    
    if( $sections ){
		 foreach ( $sections as $section ){
	        $enabled_section[] = array(
	            'id' => $section,
	            'menu_text' => esc_attr( get_theme_mod( 'business_one_page_pro_' . $section . '_section_menu_title' ) ),
	        );
	    }
    }
    return $enabled_section;  
}

/**
 * Function to list dynamic sidebar
*/
function business_one_page_pro_get_dynamnic_sidebar( $nosidebar = false, $sidebar = false, $default = false ){
    $sidebar_arr = array();
    $sidebars = get_theme_mod( 'business_one_page_pro_sidebar' );
    
    if( $default ) $sidebar_arr['default-sidebar'] = __( 'Default Sidebar', 'business-one-page-pro' );
    if( $sidebar ) $sidebar_arr['sidebar'] = __( 'Sidebar', 'business-one-page-pro' );
    
    if( $sidebars ){
        foreach( $sidebars as $sidebar ){
            $sidebar_arr[$sidebar['id']] = $sidebar['name'];
        }
    }
    
    if( $nosidebar ) $sidebar_arr['no-sidebar'] = __( 'No Sidebar', 'business-one-page-pro' );
    
    return $sidebar_arr;
}

/**
 * Function to list Custom Pattern
*/
function business_one_page_pro_get_patterns(){
    $patterns = array();
    $patterns['nobg'] = get_template_directory_uri() . '/images/patterns_thumb/' . 'nobg.png';
    for( $i=0; $i<38; $i++ ){
        $patterns['pattern'.$i] = get_template_directory_uri() . '/images/patterns_thumb/' . 'pattern' . $i .'.png';
    }
    for( $j=1; $j<26; $j++ ){
        $patterns['hbg'.$j] = get_template_directory_uri() . '/images/patterns_thumb/' . 'hbg' . $j . '.png';
    }
    return $patterns;
}

/**
 * Function to populate list of social Icons
*/
function business_one_page_pro_social_icons(){
    $social_icons = array();
    
    $social_icons['dribbble']      = __( 'Dribbble', 'business-one-page-pro' );
    $social_icons['facebook']      = __( 'Facebook', 'business-one-page-pro' );
    $social_icons['foursquare']    = __( 'Foursquare', 'business-one-page-pro' );
    $social_icons['flickr']        = __( 'Flickr', 'business-one-page-pro' );
    $social_icons['google-plus']   = __( 'Google Plus', 'business-one-page-pro' );
    $social_icons['instagram']     = __( 'Instagram', 'business-one-page-pro' );
    $social_icons['linkedin']      = __( 'LinkedIn', 'business-one-page-pro' );
    $social_icons['pinterest']     = __( 'Pinterest', 'business-one-page-pro' );
    $social_icons['reddit']        = __( 'Reddit', 'business-one-page-pro' );
    $social_icons['skype']         = __( 'Skype', 'business-one-page-pro' );
    $social_icons['stumbleupon']   = __( 'StumbleUpon', 'business-one-page-pro' );
    $social_icons['tumblr']        = __( 'Tumblr', 'business-one-page-pro' );
    $social_icons['twitter']       = __( 'Twitter', 'business-one-page-pro' );
    $social_icons['vimeo']         = __( 'Vimeo', 'business-one-page-pro' );
    $social_icons['youtube']       = __( 'YouTube', 'business-one-page-pro' );
    $social_icons['odnoklassniki'] = __( 'OK', 'business-one-page-pro' );
    $social_icons['vk']            = __( 'VK', 'business-one-page-pro' );
    $social_icons['xing']          = __( 'Xing', 'business-one-page-pro' );
    
    return $social_icons;
}

/**
 * Get an attachment ID given a URL.
 * 
 * @param string $url
 *
 * @return int Attachment ID on success, 0 on failure
 * @link https://wpscholar.com/blog/get-attachment-id-from-wp-image-url/
 */
function business_one_page_pro_get_attachment_id( $url ) {

	$attachment_id = 0;

	$dir = wp_upload_dir();

	if ( false !== strpos( $url, $dir['baseurl'] . '/' ) ) { // Is URL in uploads directory?

		$file = basename( $url );

		$query_args = array(
			'post_type'   => 'attachment',
			'post_status' => 'inherit',
			'fields'      => 'ids',
			'meta_query'  => array(
				array(
					'value'   => $file,
					'compare' => 'LIKE',
					'key'     => '_wp_attachment_metadata',
				),
			)
		);

		$query = new WP_Query( $query_args );

		if ( $query->have_posts() ) {

			foreach ( $query->posts as $post_id ) {

				$meta = wp_get_attachment_metadata( $post_id );

				$original_file       = basename( $meta['file'] );
				$cropped_image_files = wp_list_pluck( $meta['sizes'], 'file' );

				if ( $original_file === $file || in_array( $file, $cropped_image_files ) ) {
					$attachment_id = $post_id;
					break;
				}

			}

		}

	}

	return $attachment_id;
}

/**
 * List out font awesome icon list
*/
function business_one_page_pro_get_icon_list(){
    include get_template_directory() . '/inc/fontawesome.php';
    ?>
    <input type="text" class="rrtc-search-icon" placeholder="<?php _e( 'Search Icon','business-one-page-pro' );?>" />
        <div class="rara-font-awesome-list">
	    	<ul class="rara-font-group">
		    <?php
			    foreach( $fontawesome as $font ){
			        echo '<li><i class="' . esc_attr( $font ) . '"></i></li>';
			    }
		    ?>
		   </ul>
	</div>
    <?php 
}

/**
 * Helper Function for Image widget
*/
function business_one_page_pro_get_image_field( $id, $name, $image, $label ){
    
    $output = '';
    
    $output .= '<div class="widget-upload">';
    $output .= '<label for="' . esc_attr( $id ) . '">' . esc_html( $label ) . '</label><br/>';
    $output .= '<input id="' . esc_attr( $id ) . '" class="rara-upload" type="hidden" name="' . esc_attr( $name ) . '" value="' . esc_attr( $image ) . '" placeholder="' . __('No file chosen', 'business-one-page-pro') . '" />' . "\n";
    if ( function_exists( 'wp_enqueue_media' ) ) {
        if ( $image == '' ) {
            $output .= '<input id="upload-' . esc_attr( $id ) . '" class="rara-upload-button button" type="button" value="' . __('Upload', 'business-one-page-pro') . '" />' . "\n";
        } else {
            $output .= '<input id="upload-' . esc_attr( $id ) . '" class="rara-upload-button button" type="button" value="' . __('Change', 'business-one-page-pro') . '" />' . "\n";
        }
    } else {
        $output .= '<p><i>' . __('Upgrade your version of WordPress for full media support.', 'business-one-page-pro') . '</i></p>';
    }

    $output .= '<div class="rara-screenshot" id="' . esc_attr( $id ) . '-image">' . "\n";

    if ( $image != '' ) {
        $remove = '<a class="rara-remove-image"></a>';
        $attachment_id = $image;
        $image_url = wp_get_attachment_image_url( $attachment_id, 'full');
        if ( $image ) {
            $output .= '<img src="' . esc_url( $image_url ) . '" alt="" />' . $remove;
        } else {
            // Standard generic output if it's not an image.
            $output .= '<small>' . __( 'Please upload valid image file.', 'business-one-page-pro' ) . '</small>';
        }     
    }
    $output .= '</div></div>' . "\n";
    
    echo $output;
}

/**
 * Client Logo Slider
*/
function business_one_page_pro_client_logo( $home = true ){
    $title = $home ? get_theme_mod( 'business_one_page_pro_clients_section_title', __( 'Our Clients', 'business-one-page-pro' ) ) : get_theme_mod( 'business_one_page_pro_our_client_title', __( 'Our Clients', 'business-one-page-pro' ) );
    $bg    = $home ? get_theme_mod( 'business_one_page_pro_clients_section_bg', get_template_directory_uri() . '/images/img11.jpg' ) : get_theme_mod( 'business_one_page_pro_our_clients_bg', get_template_directory_uri() . '/images/img11.jpg' );
    $client_qry = new WP_Query( array( 'post_type' => 'logo', 'posts_per_page' => -1, 'post_status' => 'publish' ) );
    
    if( $client_qry->have_posts() || $title ){  
    ?>
    <section id="clients" class="our-clients" <?php if( $bg ) echo 'style="background: url(' . esc_url( $bg ) .') no-repeat; background-size: cover;"' ?> >
    	<div class="container">
    		<?php 
                
                if( $title ) echo '<h2 class="title">' . esc_html( $title ) . '</h2>'; 
                
                if( $client_qry->have_posts() ){        
                ?>
                    <div id="client-slider" class="owl-carousel owl-theme">
                    <?php
                        while( $client_qry->have_posts() ){
                            $client_qry->the_post();
                            
                            $link = get_post_meta( get_the_ID(), '_business_one_page_pro_logo_link', true );
                            echo '<div>';
                            if( $link ) echo '<a href="' . esc_url( $link ) . '" target="_blank" title="' . esc_attr( get_the_title() ) . '">';
                            
                            the_post_thumbnail( 'full', array( 'itemprop' => 'image' ) );
                            
                            if( $link ) echo '</a>';
                            
                            echo '</div>';        
                        }
                    ?>
                    </div>    
                <?php
                }
                ?>
    	</div>
    </section>
    <?php
    }    
}

/**
 *  Custom Pagination
*/
function business_one_page_pro_pagination(){
    $pagination = get_theme_mod( 'business_one_page_pro_pagination_type', 'default' );
    
    switch( $pagination ){
        case 'default': // Default Pagination
        
        the_posts_navigation();
        
        break;
        
        case 'numbered': // Numbered Pagination
        
        the_posts_pagination( array(
            'prev_text'          => __( 'Previous', 'business-one-page-pro' ),
            'next_text'          => __( 'Next', 'business-one-page-pro' ),
            'before_page_number' => '<span class="meta-nav screen-reader-text">' . __( 'Page', 'business-one-page-pro' ) . ' </span>',
         ) );
        
        break;
        
        case 'load_more': // Load More Button
        case 'infinite_scroll': // Auto Infinite Scroll
        
        echo '<div class="pagination"></div>';
        
        break;
        
        default:
        
        the_posts_navigation();
        
        break;
    }   
}

/**
 * Function to retrive page specific sidebar and corresponding body class
 * 
 * @param boolean $sidebar
 * @param boolean $class
 * 
 * @return string dynamic sidebar id / classname
*/
function business_one_page_pro_sidebar( $sidebar = false, $class = false ){
    
    global $post;
    $return = false;
    
    if( ( is_front_page() && is_home() ) || is_home() ){
        //blog/home page 
        $blog_sidebar = get_theme_mod( 'business_one_page_pro_blog_page_sidebar', 'sidebar' );
        $blog_layout  = get_theme_mod( 'business_one_page_pro_blog_layout', 'default' );
        $layout       = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( is_active_sidebar( $blog_sidebar ) ){            
            if( $sidebar ) $return = $blog_sidebar; //With Sidebar
            if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; 
            if( $class && $layout == 'left-sidebar' )  $return = 'leftsidebar';    
            
            if( $class ){
                if( $blog_layout == 'full' ){
                    $return .= ' blog-large';
                }elseif( $blog_layout == 'square' ){
                    $return .= ' blog-medium';
                }elseif( $blog_layout == 'circular'  ){
                    $return .= ' blog-round';
                }        
            }
        }else{
            if( $sidebar ) $return = false; //Fullwidth
            if( $class ) $return = 'full-width';
            
            if( $class ){
                if( $blog_layout == 'full' ){
                    $return .= ' blog-large';
                }elseif( $blog_layout == 'square' ){
                    $return .= ' blog-medium';
                }elseif( $blog_layout == 'circular'  ){
                    $return .= ' blog-round';
                }
            }
        }        
    }
    
    if( is_archive() ){
        //archive page
        $archive_sidebar = get_theme_mod( 'business_one_page_pro_archive_page_sidebar', 'sidebar' );
        $cat_sidebar     = get_theme_mod( 'business_one_page_pro_cat_archive_page_sidebar', 'default-sidebar' );
        $tag_sidebar     = get_theme_mod( 'business_one_page_pro_tag_archive_page_sidebar', 'default-sidebar' );
        $date_sidebar    = get_theme_mod( 'business_one_page_pro_date_archive_page_sidebar', 'default-sidebar' );
        $author_sidebar  = get_theme_mod( 'business_one_page_pro_author_archive_page_sidebar', 'default-sidebar' );
        $layout          = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( is_category() ){
            
            if( $cat_sidebar == 'no-sidebar' || ( $cat_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( $cat_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $cat_sidebar ) ){
                if( $sidebar ) $return = $cat_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
                
        }elseif( is_tag() ){
            
            if( $tag_sidebar == 'no-sidebar' || ( $tag_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( ( $tag_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $tag_sidebar ) ){
                if( $sidebar ) $return = $tag_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';              
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
            
        }elseif( is_author() ){
            
            if( $author_sidebar == 'no-sidebar' || ( $author_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( ( $author_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $author_sidebar ) ){
                if( $sidebar ) $return = $author_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
            
        }elseif( is_date() ){
            
            if( $date_sidebar == 'no-sidebar' || ( $date_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( ( $date_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $date_sidebar ) ){
                if( $sidebar ) $return = $date_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }                         
            
        }elseif( is_woocommerce_activated() && ( is_shop() || is_product_category() || is_product_tag() ) ){ //For Woocommerce
            
            if( is_active_sidebar( 'shop-sidebar' ) ){
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                if( $class ) $return = 'full-width';
            }
                   
        }else{
            if( $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }                      
        }
        
    }
    
    if( is_singular() ){
        $post_sidebar = get_theme_mod( 'business_one_page_pro_single_post_sidebar', 'sidebar' );
        $page_sidebar = get_theme_mod( 'business_one_page_pro_single_page_sidebar', 'sidebar' );
        $layout       = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( get_post_meta( $post->ID, '_business_one_page_pro_sidebar', true ) ){
            $single_sidebar = get_post_meta( $post->ID, '_business_one_page_pro_sidebar', true );
        }else{
            $single_sidebar = 'default-sidebar';
        }

        if( get_post_meta( $post->ID, '_business_one_page_pro_sidebar_layout', true ) ){
            $sidebar_layout = get_post_meta( $post->ID, '_business_one_page_pro_sidebar_layout', true );
        }else{
            $sidebar_layout = 'default-sidebar';
        }
        
        if( is_page() ){
            
            if( is_page_template( 'templates/template-home.php' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }else{
                if( ( $single_sidebar == 'no-sidebar' ) || ( ( $single_sidebar == 'default-sidebar' ) && ( $page_sidebar == 'no-sidebar' ) ) ){
                    if( $sidebar ) $return = false; //Fullwidth
                    if( $class ) $return = 'full-width';
                }elseif( $single_sidebar == 'default-sidebar' && $page_sidebar != 'no-sidebar' && is_active_sidebar( $page_sidebar ) ){
                    if( $sidebar ) $return = $page_sidebar;
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
                }elseif( is_active_sidebar( $single_sidebar ) ){
                    if( $sidebar ) $return = $single_sidebar;
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
                }else{
                    if( $sidebar ) $return = false; //Fullwidth
                    if( $class ) $return = 'full-width';
                }
            }
        }
        
        if( is_single() ){
            
            if( is_woocommerce_activated() && 'product' === get_post_type() ){
                if( is_active_sidebar( 'shop-sidebar' ) ){
                    if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                    if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
                }else{
                    if( $class ) $return = 'full-width';
                }
            }elseif( ( $single_sidebar == 'no-sidebar' ) || ( ( $single_sidebar == 'default-sidebar' ) && ( $post_sidebar == 'no-sidebar' ) ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( $single_sidebar == 'default-sidebar' && $post_sidebar != 'no-sidebar' && is_active_sidebar( $post_sidebar ) ){
                if( $sidebar ) $return = $post_sidebar;
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $single_sidebar ) ){
                if( $sidebar ) $return = $single_sidebar;
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
        }
    }
    
    if( is_search() ){
        $search_sidebar = get_theme_mod( 'business_one_page_pro_search_page_sidebar', 'sidebar' );
        $layout         = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( $search_sidebar != 'no-sidebar' && is_active_sidebar( $search_sidebar ) ){
            if( $sidebar ) $return = $search_sidebar;
            if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
            if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
        }else{
            if( $sidebar ) $return = false; //Fullwidth
            if( $class ) $return = 'full-width';
        }
        
    }
    
    return $return;        
}

/**
 * Function to get the post view count 
 */
function business_one_page_pro_get_views( $post_id ){
    $count_key = '_business_one_page_pro_view_count';
    $count = get_post_meta( $post_id, $count_key, true );
    if( $count == '' ){        
        return __( "0 View", 'business-one-page-pro' );
    }elseif($count<=1){
        return $count. __(' View', 'business-one-page-pro' );
    }else{
        return $count. __(' Views', 'business-one-page-pro' );    
    }    
}

/**
 * Function to add the post view count 
 */
function business_one_page_pro_set_views( $post_id ) {
    $count_key = '_business_one_page_pro_view_count';
    $count = get_post_meta( $post_id, $count_key, true );
    if( $count == '' ){
        $count = 0;
        delete_post_meta( $post_id, $count_key );
        add_post_meta( $post_id, $count_key, '1' );
    }else{
        $count++;
        update_post_meta( $post_id, $count_key, $count );
    }
}

/**
 * Helper function for strip specific html tags from a string
 * @link http://www.altafweb.com/2011/12/remove-specific-tag-from-php-string.html
*/
function business_one_page_pro_strip_single( $tag, $string ){
    $string=preg_replace('/<'.$tag.'[^>]*>/i', '', $string);
    $string=preg_replace('/<\/'.$tag.'>/i', '', $string);
    return $string;
}

/**
 * convert hex to rgb
 * @link http://bavotasan.com/2011/convert-hex-color-to-rgb-using-php/
*/
function business_one_page_pro_hex2rgb($hex) {
   $hex = str_replace("#", "", $hex);

   if(strlen($hex) == 3) {
      $r = hexdec(substr($hex,0,1).substr($hex,0,1));
      $g = hexdec(substr($hex,1,1).substr($hex,1,1));
      $b = hexdec(substr($hex,2,1).substr($hex,2,1));
   } else {
      $r = hexdec(substr($hex,0,2));
      $g = hexdec(substr($hex,2,2));
      $b = hexdec(substr($hex,4,2));
   }
   $rgb = array($r, $g, $b);
   //return implode(",", $rgb); // returns the rgb values separated by commas
   return $rgb; // returns an array with the rgb values
}

if ( ! function_exists( 'business_one_page_pro_apply_theme_shortcode' ) ) :
/**
 * Footer Shortcode
*/
function business_one_page_pro_apply_theme_shortcode( $string ) {
    if ( empty( $string ) ) {
        return $string;
    }
    $search = array( '[the-year]', '[the-site-link]' );
    $replace = array(
        date_i18n( esc_html__( 'Y', 'business-one-page-pro' ) ),
        '<a href="'. esc_url( home_url( '/' ) ) .'">'. esc_html( get_bloginfo( 'name', 'display' ) ) . '</a>',
    );
    $string = str_replace( $search, $replace, $string );
    return $string;
}
endif;

/**
 * Query WooCommerce activation
 */
function is_woocommerce_activated() {
	return class_exists( 'woocommerce' ) ? true : false;
}

/**
 * Query Contact Form 7
 */
function is_cf7_activated() {
	return class_exists( 'WPCF7' ) ? true : false;
}

/**
 * Query Jetpack activation
*/
function is_jetpack_activated( $gallery = false ){
	if( $gallery ){
        return ( class_exists( 'jetpack' ) && Jetpack::is_module_active( 'tiled-gallery' ) ) ? true : false;
	}else{
        return class_exists( 'jetpack' ) ? true : false;
    }           
}

/**
 * Query if Rara One Click Demo Import id activate
*/
function is_rocdi_activated(){
    return class_exists( 'RDDI_init' ) ? true : false;
}

/**
 * Check if Polylang is active
*/
function is_polylang_active(){
    return class_exists( 'Polylang' ) ? true : false;
}

function is_elementor_activated(){
    return class_exists( 'Elementor\\Plugin' ) ? true : false; 
}

function is_elementor_activated_post(){
    if( is_elementor_activated() ){
    global $post;
    $post_id = $post->ID;
        return \Elementor\Plugin::$instance->db->is_built_with_elementor( $post_id ) ? true : false;
    }else{
        return false;
    }
}

if( ! function_exists( 'business_one_page_pro_escape_text_tags' ) ) :
/**
 * Remove new line tags from string
 *
 * @param $text
 *
 * @return string
 */
function business_one_page_pro_escape_text_tags( $text ) {
    return (string) str_replace( array( "\r", "\n" ), '', strip_tags( $text ) );
}
endif;

if ( ! function_exists( 'business_one_page_pro_slider_animation_options' ) ) : 
    function business_one_page_pro_slider_animation_options(){
        $slider_animation = array(
            'slide'          => esc_html__( 'Slide' , 'business-one-page-pro' ),
            'fade'           => esc_html__( 'Fade' , 'business-one-page-pro' ),
            'bounceOut'      => esc_html__( 'Bounce Out', 'business-one-page-pro' ),
            'bounceOutLeft'  => esc_html__( 'Bounce Out Left', 'business-one-page-pro' ),
            'bounceOutRight' => esc_html__( 'Bounce Out Right', 'business-one-page-pro' ),
            'bounceOutUp'    => esc_html__( 'Bounce Out Up', 'business-one-page-pro' ),
            'bounceOutDown'  => esc_html__( 'Bounce Out Down', 'business-one-page-pro' ),
            'fadeOut'        => esc_html__( 'Fade Out', 'business-one-page-pro' ),
            'fadeOutLeft'    => esc_html__( 'Fade Out Left', 'business-one-page-pro' ),
            'fadeOutRight'   => esc_html__( 'Fade Out Right', 'business-one-page-pro' ),
            'fadeOutUp'      => esc_html__( 'Fade Out Up', 'business-one-page-pro' ),
            'fadeOutDown'    => esc_html__( 'Fade Out Down', 'business-one-page-pro' ),
            'flipOutX'       => esc_html__( 'Flip OutX', 'business-one-page-pro' ),
            'flipOutY'       => esc_html__( 'Flip OutY', 'business-one-page-pro' ),
            'hinge'          => esc_html__( 'Hinge', 'business-one-page-pro' ),
            'pulse'          => esc_html__( 'Pulse', 'business-one-page-pro' ),
            'rollOut'        => esc_html__( 'Roll Out', 'business-one-page-pro' ),
            'rotateOut'      => esc_html__( 'Rotate Out', 'business-one-page-pro' ),
            'rotateOutLeft'  => esc_html__( 'Rotate Out Left', 'business-one-page-pro' ),
            'rotateOutRight' => esc_html__( 'Rotate Out Right', 'business-one-page-pro' ),
            'rotateOutUp'    => esc_html__( 'Rotate Out Up', 'business-one-page-pro' ),
            'rotateOutDown'  => esc_html__( 'Rotate Out Down', 'business-one-page-pro' ),
            'rubberBand'     => esc_html__( 'Rubber Band', 'business-one-page-pro' ),
            'shake'          => esc_html__( 'Shake', 'business-one-page-pro' ),
            'slideOutLeft'   => esc_html__( 'Slide Out Left', 'business-one-page-pro' ),
            'slideOutRight'  => esc_html__( 'Slide Out Right', 'business-one-page-pro' ),
            'slideOutUp'     => esc_html__( 'Slide Out Up', 'business-one-page-pro' ),
            'slideOutDown'   => esc_html__( 'Slide Out Down', 'business-one-page-pro' ),
            'swing'          => esc_html__( 'Swing', 'business-one-page-pro' ),
            'tada'           => esc_html__( 'Tada', 'business-one-page-pro' ),
            'zoomOut'        => esc_html__( 'Zoom Out', 'business-one-page-pro' ),
            'zoomOutLeft'    => esc_html__( 'Zoom Out Left', 'business-one-page-pro' ),
            'zoomOutRight'   => esc_html__( 'Zoom Out Right', 'business-one-page-pro' ),
            'zoomOutUp'      => esc_html__( 'Zoom Out Up', 'business-one-page-pro' ),
            'zoomOutDown'    => esc_html__( 'Zoom Out Down', 'business-one-page-pro' ),
        );

        return apply_filters( 'business_one_page_pro_slider_animation_options', $slider_animation ); 
    }
endif;

if( ! function_exists( 'business_one_page_pro_get_page_template_url' ) ) :
/**
 * Returns page template url if not found returns home page url
*/
function business_one_page_pro_get_page_template_url( $page_template ){
    $args = array(
        'meta_key'   => '_wp_page_template',
        'meta_value' => $page_template,
        'post_type'  => 'page',
        'fields'     => 'ids',
    );
    
    $posts_array = get_posts( $args );
        
    $url = ( $posts_array ) ? get_permalink( $posts_array[0] ) : get_home_url();
    return $url;    
}
endif;

if( ! function_exists( 'business_one_page_pro_get_blog_page_url' ) ) :
/**
 * Returns page url of static blog page if not found returns home page url
*/
function business_one_page_pro_get_blog_page_url(){
	$show_on_front    = get_option( 'show_on_front' );
	$blog_page_id     = get_option( 'page_for_posts' );

	if( $show_on_front = 'page' ){
		if( $blog_page_id ){
	        return get_the_permalink( $blog_page_id );
	    }else{
	        return get_home_url();
	    }	
	}else{
        return get_home_url();
	} 
}
endif;

if( ! function_exists( 'business_one_page_pro_polylang_language_switcher' ) ) :
    /**
     * Template for Polylang Language Switcher
     */
    function business_one_page_pro_polylang_language_switcher(){
        if( is_polylang_active() && has_nav_menu( 'language' ) ){ ?>
            <nav class="language-dropdown">
                <?php
                    wp_nav_menu( array(
                        'theme_location' => 'language',
                        'menu_class'     => 'languages',
                        'fallback_cb'    => false,
                    ) );
                ?>
            </nav><!-- #site-navigation -->
            <?php        
        }
    }
endif;

if( ! function_exists( 'wp_body_open' ) ) :
/**
 * Fire the wp_body_open action.
 * Added for backwards compatibility to support pre 5.2.0 WordPress versions.
*/
function wp_body_open() {
	/**
	 * Triggered after the opening <body> tag.
    */
	do_action( 'wp_body_open' );
}
endif;

if( ! function_exists( 'business_one_page_pro_duplicate_post' ) ) :
/**
 * Duplicate normal post to custom post on theme switch.
 * 
 * @param post id, post type
 * @return post id
*/
function business_one_page_pro_duplicate_post( $post_id, $post_type, $attachment = false ){
    if( $post_id ){    
        $post         = get_post( absint( $post_id ) );
        $thumbnail_id = $attachment ? $post_id : get_post_meta( $post_id, '_thumbnail_id', true );
        
        if( isset( $post ) && $post != null ){
            /** new post data array */
    		$args = array(
    			'post_author'   => $post->post_author,
    			'post_content'  => $post->post_content,
    			'post_excerpt'  => $post->post_excerpt,
    			'post_name'     => $post->post_name,
    			'post_password' => $post->post_password,
    			'post_status'   => 'publish',
    			'post_title'    => $post->post_title,
    			'post_type'     => $post_type,
    		);
     
    		/** insert the post by wp_insert_post() function */
    		$new_post_id = wp_insert_post( $args );
            if( $new_post_id ){
                /** If post created then set post thumbnail of old post */
                set_post_thumbnail( $new_post_id, $thumbnail_id );
                return $new_post_id;                
            }else{
                return false;
            }
        }else{
    		wp_die( 'Post creation failed, could not find original post: ' . $post_id );
    	}
    }else{
        return false;
    }
}
endif;

if( ! function_exists( 'business_one_page_pro_create_post' ) ) :
/**
 * A function used to programmatically create a post and assign a page template in WordPress. 
 *
 * @link https://tommcfarlin.com/programmatically-create-a-post-in-wordpress/
 * @link https://tommcfarlin.com/programmatically-set-a-wordpress-template/
 */
function business_one_page_pro_create_post( $title, $slug, $template ){

	// Setup the author, page
	$author_id = 1;
    
    // Look for the page by the specified title. Set the ID to -1 if it doesn't exist.
    // Otherwise, set it to the page's ID.
    $page = get_page_by_title( $title, OBJECT, 'page' );
    $page_id = ( null == $page ) ? -1 : $page->ID;
    
	// If the page doesn't already exist, then create it
	if( $page_id == -1 ){

		// Set the post ID so that we know the post was created successfully
		$post_id = wp_insert_post(
			array(
				'comment_status' =>	'closed',
				'ping_status'	 =>	'closed',
				'post_author'	 =>	$author_id,
				'post_name'		 =>	$slug,
				'post_title'	 =>	$title,
				'post_status'	 =>	'publish',
				'post_type'		 =>	'page'
			)
		);
        
        if( $post_id ) update_post_meta( $post_id, '_wp_page_template', $template );

	// Otherwise, we'll stop
	}else{
	   update_post_meta( $page_id, '_wp_page_template', $template );
	} // end if

} // end programmatically_create_post
endif;