<?php
/**
 * Custom Functions used in this theme.
 *
 * @package Business_One_Page_Pro
 */

if( ! get_theme_mod( 'business_one_page_pro_ed_adminbar' ) ) add_filter( 'show_admin_bar', '__return_false' );
 
if ( ! function_exists( 'business_one_page_pro_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function business_one_page_pro_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Business One Page, use a find and replace
	 * to change 'business-one-page-pro' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'business-one-page-pro', get_template_directory() . '/languages' );
    
	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
    $menus = array(
        'primary' => esc_html__( 'Primary', 'business-one-page-pro' ),
        'top'     => esc_html__( 'Top', 'business-one-page-pro' ),
    );

    if( is_polylang_active() ){
        $menus['language'] = esc_html__( 'Language', 'business-one-page-pro' ); 
    }
    
	register_nav_menus( $menus );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	/*
	 * Enable support for Post Formats.
	 * See https://developer.wordpress.org/themes/functionality/post-formats/
	 */
	add_theme_support( 'post-formats', array(
		'aside',
		'image',
		'video',
		'quote',
		'link',
	) );

	/* Custom Logo */
    add_theme_support( 'custom-logo', array(
    	'header-text' => array( 'site-title', 'site-description' ),
    ) );
    
    //Custom Image Sizes
    add_image_size( 'business-one-page-pro-post-thumb', 80, 70, true);    
    add_image_size( 'business-one-page-pro-slider', 1400, 577, true);
    
    add_image_size( 'business-one-page-pro-full', 1170, 480, true);
    add_image_size( 'business-one-page-pro-with-sidebar', 750, 460, true);
    add_image_size( 'business-one-page-pro-cat-blog', 750, 360, true);
    add_image_size( 'business-one-page-pro-blog-small', 293, 293, true);
    add_image_size( 'business-one-page-pro-team', 340, 310, true);
    
    add_image_size( 'business-one-page-pro-blog', 360, 280, true);
    add_image_size( 'business-one-page-pro-testimonial', 111, 111, true);
    add_image_size( 'business-one-page-pro-360x340', 360, 340, true);
    add_image_size( 'business-one-page-pro-360x380', 360, 380, true);
    add_image_size( 'business-one-page-pro-360x500', 360, 500, true);
    add_image_size( 'business-one-page-pro-schema', 600, 60, true);
    
    // Add theme support for selective refresh for widgets.
    add_theme_support( 'customize-selective-refresh-widgets' );   
    
    /** Starter Content */
    $starter_content = array(
        // Specify the core-defined pages to create and add custom thumbnails to some of them.
        'posts' => array( 
            'homepage' => array(
                'post_type'  => 'page',
                'post_title' => esc_html__( 'Homepage', 'business-one-page-pro' ),
                'template'   => 'templates/template-home.php',
            ), 
            'blog',
            'about-us' => array(
                'post_type'  => 'page',
                'post_title' => esc_html__( 'About Us', 'business-one-page-pro' ),
                'template'   => 'templates/template-about.php',
            ),
            'contact-us' => array(
                'post_type'  => 'page',
                'post_title' => esc_html__( 'Contact Us', 'business-one-page-pro' ),
                'template'   => 'templates/template-contact.php',
            ),
            'our-portfolio' => array(
                'post_type'  => 'page',
                'post_title' => esc_html__( 'Our Portfolio', 'business-one-page-pro' ),
                'template'   => 'templates/template-portfolio.php',
            ),
            'our-team' => array(
                'post_type'  => 'page',
                'post_title' => esc_html__( 'Our Team', 'business-one-page-pro' ),
                'template'   => 'templates/template-team.php',
            ),
            'testimonials' => array(
                'post_type'  => 'page',
                'post_title' => esc_html__( 'Testimonials', 'business-one-page-pro' ),
                'template'   => 'templates/template-testimonial.php',
            ),
            'services' => array(
                'post_type'  => 'page',
                'post_title' => esc_html__( 'Services', 'business-one-page-pro' ),
                'template'   => 'templates/template-services.php',
            ), 
        ),
        
        // Default to a static front page and assign the front and posts pages.
        'options' => array(
            'show_on_front' => 'page',
            'page_on_front' => '{{homepage}}',
            'page_for_posts' => '{{blog}}',
        ),
        
        // Set up nav menus for each of the two areas registered in the theme.
        'nav_menus' => array(
            // Assign a menu to the "top" location.
            'top' => array(
                'name' => __( 'Top Header Menu', 'business-one-page-pro' ),
                'items' => array(                    
                    'page_about_us' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{about-us}}',
                    ),
                    'page_services' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{services}}',
                    ),
                    'page_our_portfolio' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{our-portfolio}}',
                    ),
                    'page_testimonials' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{testimonials}}',
                    ),
                    'page_our_team' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{our-team}}',
                    ),                    
                    'page_blog',
                    'page_contact_us' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{contact-us}}',
                    ),
                )
            )
        ),
    );

    $starter_content = apply_filters( 'business_one_page_pro_starter_content', $starter_content );

    add_theme_support( 'starter-content', $starter_content );

}
endif;
add_action( 'after_setup_theme', 'business_one_page_pro_setup' );

if( ! function_exists( 'business_one_page_pro_content_width' ) ) :
/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function business_one_page_pro_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'business_one_page_pro_content_width', 750 );
}
endif;
add_action( 'after_setup_theme', 'business_one_page_pro_content_width', 0 );

if( ! function_exists( 'business_one_page_pro_scripts' ) ) :
/**
 * Enqueue scripts and styles.
 */
function business_one_page_pro_scripts(){
	
    $build  = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '/build' : '';
    $suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';
    
    $map_api = get_theme_mod( 'business_one_page_pro_map_api' );
    
    $key = $map_api ? '?key='.esc_attr( $map_api ) : '' ;
    $ed_elementor = get_theme_mod( 'ed_elementor', '0' );
    wp_enqueue_style( 'owl-theme-default', get_template_directory_uri(). '/css' . $build . '/owl.theme.default' . $suffix . '.css', array(), '2.2.1' );

    wp_enqueue_style( 'owl-carousel', get_template_directory_uri(). '/css' . $build . '/owl.carousel' . $suffix . '.css', array(), '2.2.1' );
    wp_enqueue_style( 'animate', get_template_directory_uri() . '/css' . $build . '/animate' . $suffix . '.css', array(), '3.5.2' );
    wp_enqueue_style( 'jquery-scrollbar', get_template_directory_uri() . '/css' . $build . '/jquery.mCustomScrollbar' . $suffix . '.css', array(), '3.1.5' );
    wp_enqueue_style( 'business-one-page-pro-style', get_stylesheet_uri(), '', BUSINESS_ONE_PAGE_PRO_THEME_VERSION );
    
    if( is_elementor_activated() && $ed_elementor ){
        wp_enqueue_style( 'business-one-page-pro-elementor', get_template_directory_uri(). '/css/' . $build . 'elementor' . $suffix . '.css', array(), BUSINESS_ONE_PAGE_PRO_THEME_VERSION );
    }
    
    if( is_woocommerce_activated() )
    wp_enqueue_style( 'business-one-page-pro-woocommerce-style', get_template_directory_uri(). '/css' . $build . '/woocommerce' . $suffix . '.css', array('business-one-page-pro-style'), BUSINESS_ONE_PAGE_PRO_THEME_VERSION );
    
    if( get_theme_mod( 'business_one_page_pro_ed_lightbox') ){
        wp_enqueue_style( 'jquery-fancybox', get_template_directory_uri() . '/css' . $build . '/jquery.fancybox' . $suffix . '.css', '', '3.5.6' );
        wp_enqueue_script( 'jquery-fancybox', get_template_directory_uri() . '/js' . $build . '/jquery.fancybox' . $suffix . '.js', array('jquery'), '3.5.6', true );
    }
    
    wp_enqueue_script( 'all', get_template_directory_uri() . '/js' . $build . '/all' . $suffix . '.js', array( 'jquery' ), '5.6.3', false );

    wp_enqueue_script( 'v4-shims', get_template_directory_uri() . '/js' . $build . '/v4-shims' . $suffix . '.js', array( 'jquery' ), '5.6.3', false );

	if( is_page_template( 'templates/template-contact.php' ) )
    wp_enqueue_script( 'business-one-page-pro-googlemap', '//maps.googleapis.com/maps/api/js'.$key, array('jquery'), '3.0', false );
    
    if( is_page_template( array( 'templates/template-home.php', 'templates/template-portfolio.php', 'templates/template-testimonial.php', 'templates/template-team.php' ) ) )
    wp_enqueue_script( 'masonry' );
    
    if( is_page_template( 'templates/template-portfolio.php' ) )
    wp_enqueue_script( 'isotope', get_template_directory_uri() . '/js' . $build . '/isotope.pkgd' . $suffix . '.js', array('jquery'), '3.0.1', true );
    
    if( is_active_widget( false, false, 'business_one_page_pro_flickr_widget' ) )
    wp_enqueue_script( 'flickr', get_template_directory_uri() . '/js' . $build . '/jflickrfeed' . $suffix . '.js', array('jquery'), BUSINESS_ONE_PAGE_PRO_THEME_VERSION, true );
    
    if( is_active_widget( false, false, 'business_one_page_pro_stat_counter_widget' ) ){
        wp_enqueue_script( 'waypoint', get_template_directory_uri() . '/js' . $build . '/waypoint' . $suffix . '.js', array('jquery'), '2.0.3', true );
        wp_enqueue_script( 'counterup', get_template_directory_uri() . '/js' . $build . '/jquery.counterup' . $suffix . '.js', array('jquery', 'waypoint'), '1.0', true );
    }

    if( get_theme_mod( 'ed_lazy_load', false ) || get_theme_mod( 'ed_lazy_load_cimage', false ) ) {
        wp_enqueue_script( 'layzr', get_template_directory_uri() . '/js' . $build . '/layzr' . $suffix . '.js', array('jquery'), '2.0.4', true );
    }
    
    if( ! get_theme_mod( 'business_one_page_pro_ed_section_menu' ) )
    wp_enqueue_script( 'scrollnav', get_template_directory_uri() . '/js' . $build . '/scroll-nav' . $suffix . '.js', array('jquery'), '3.0.0', true );
    
    wp_enqueue_script( 'jquery-scrollbar', get_template_directory_uri() . '/js' . $build . '/jquery.mCustomScrollbar' . $suffix . '.js', array('jquery'), '3.1.5', true );
    wp_enqueue_script( 'fitvids', get_template_directory_uri() . '/js' . $build . '/jquery.fitvids' . $suffix . '.js', array('jquery'), '1.1', true );
    wp_enqueue_script( 'owl-carousel', get_template_directory_uri() . '/js' . $build . '/owl.carousel' . $suffix . '.js', array( 'jquery' ), '2.2.1', true );
    wp_enqueue_script( 'headroom', get_template_directory_uri() . '/js' . $build . '/headroom' . $suffix . '.js', array('jquery'), '0.7.0', true );
    wp_enqueue_script( 'equalheight', get_template_directory_uri() . '/js' . $build . '/equal-height' . $suffix . '.js', array('jquery'), '0.7.0', true );
    wp_register_script( 'business-one-page-pro-custom', get_template_directory_uri() . '/js' . $build . '/custom' . $suffix . '.js', array('jquery'), BUSINESS_ONE_PAGE_PRO_THEME_VERSION, true );
    wp_register_script( 'business-one-page-pro-ajax', get_template_directory_uri() . '/js' . $build . '/ajax' . $suffix . '.js', array('jquery'), BUSINESS_ONE_PAGE_PRO_THEME_VERSION, true );
    
    $slider_auto      = get_theme_mod( 'business_one_page_pro_slider_auto', '1' );
    $slider_loop      = get_theme_mod( 'business_one_page_pro_slider_loop', '1' );
    $slider_pager     = get_theme_mod( 'business_one_page_pro_slider_pager', '1' );    
    $slider_animation = get_theme_mod( 'business_one_page_pro_slider_animation', 'slide' );
    $slider_speed     = get_theme_mod( 'business_one_page_pro_slider_speed', '400' );
    $slider_pause     = get_theme_mod( 'business_one_page_pro_slider_pause', '6000' );
    $tslider_auto     = get_theme_mod( 'business_one_page_pro_team_slider_auto' );
    $tslider_loop     = get_theme_mod( 'business_one_page_pro_team_slider_loop','1' );
    $cslider_auto     = get_theme_mod( 'business_one_page_pro_client_slider_auto' );
    $teslider_auto    = get_theme_mod( 'business_one_page_pro_testimonial_slider_auto' );
    $te_slider_pause  = get_theme_mod( 'business_one_page_pro_testimonial_slider_pause', '4000' );
    $pause_on_hover   = get_theme_mod( 'business_one_page_pro_testimonial_slider_hover_on_pause' );

    
    $array = array(
        'auto'         => esc_attr( $slider_auto ),
        'loop'         => esc_attr( $slider_loop ),
        'pager'        => esc_attr( $slider_pager ),
        'mode'         => esc_attr( $slider_animation ),
        'speed'        => absint( $slider_speed ),
        'pause'        => absint( $slider_pause ),
        'lightbox'     => esc_attr( get_theme_mod( 'business_one_page_pro_ed_lightbox') ),
        'rtl'          => is_rtl(),
        'tauto'        => esc_attr( $tslider_auto ),
        'tloop'        => esc_attr( $tslider_loop ),
        'cauto'        => esc_attr( $cslider_auto ),
        'teauto'       => esc_attr( $teslider_auto ),
        'tepause'      => esc_attr( $te_slider_pause ),
        'pauseonhover' => esc_attr( $pause_on_hover ),
        'one_page'     => get_theme_mod( 'business_one_page_pro_ed_section_menu' ) 
    );
    
    wp_localize_script( 'business-one-page-pro-custom', 'business_one_page_pro_data', $array );
    wp_enqueue_script( 'business-one-page-pro-custom' );
    
    $pagination = get_theme_mod( 'business_one_page_pro_pagination_type', 'default' );
    $loadmore   = get_theme_mod( 'business_one_page_pro_load_more_label', __( 'Load More Posts', 'business-one-page-pro' ) );
    $loading    = get_theme_mod( 'business_one_page_pro_loading_label', __( 'Loading...', 'business-one-page-pro' ) );
    
    if( get_theme_mod( 'business_one_page_pro_ed_ajax_search' ) || $pagination == 'load_more' || $pagination == 'infinite_scroll' ){
        
        // Add parameters for the JS
        global $wp_query;
        $max = $wp_query->max_num_pages;
        $paged = ( get_query_var( 'paged' ) > 1 ) ? get_query_var( 'paged' ) : 1;
        
        wp_enqueue_script( 'business-one-page-pro-ajax' );
        
        wp_localize_script( 
            'business-one-page-pro-ajax', 
            'business_one_page_pro_ajax',
            array(
                'url'           => admin_url( 'admin-ajax.php' ),
                'startPage'     => $paged,
                'maxPages'      => $max,
                'nextLink'      => next_posts( $max, false ),
                'autoLoad'      => $pagination,
                'loadmore'      => $loadmore,
                'loading'       => $loading,
                'nomore'        => __( 'No more posts.', 'business-one-page-pro' ),
                'plugin_url'    => plugins_url()
             )
        );
        
        if ( is_jetpack_activated( true ) ) {
            wp_enqueue_style( 'tiled-gallery', plugins_url() . '/jetpack/modules/tiled-gallery/tiled-gallery/tiled-gallery.css' );            
        }
    }
    
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
endif;
add_action( 'wp_enqueue_scripts', 'business_one_page_pro_scripts' );

if( ! function_exists( 'business_one_page_pro_scripts_styles' ) ) :
/**
 * Enqueue google fonts
*/
function business_one_page_pro_scripts_styles() {
    wp_enqueue_style( 'business-one-page-pro-google-fonts', business_one_page_pro_fonts_url(), array(), null );
}
endif;
add_action( 'wp_enqueue_scripts', 'business_one_page_pro_scripts_styles' );

if( ! function_exists( 'business_one_page_pro_admin_scripts' ) ) :
/**
 * Enqueue admin scripts and styles.
 */
function business_one_page_pro_admin_scripts(){
    
    $build  = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '/build' : '';
    $suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';
    
    if( function_exists( 'wp_enqueue_media' ) )
    wp_enqueue_media();
    
    wp_enqueue_style( 'wp-color-picker' );  

    wp_enqueue_script( 'all', get_template_directory_uri() . '/js' . $build . '/all' . $suffix . '.js', array( 'jquery' ), '5.6.3', false );

    wp_enqueue_script( 'v4-shims', get_template_directory_uri() . '/js' . $build . '/v4-shims' . $suffix . '.js', array( 'jquery' ), '5.6.3', false );
    
    wp_enqueue_style( 'business-one-page-pro-admin-style', get_template_directory_uri() . '/inc/css/admin.css', array(), BUSINESS_ONE_PAGE_PRO_THEME_VERSION );
    
    wp_enqueue_script( 'business-one-page-pro-media-uploader', get_template_directory_uri() . '/inc/js/media-uploader.js', array('jquery'), BUSINESS_ONE_PAGE_PRO_THEME_VERSION );
    wp_enqueue_script( 'business-one-page-pro-admin-js', get_template_directory_uri() . '/inc/js/admin.js', array('jquery'), BUSINESS_ONE_PAGE_PRO_THEME_VERSION );
    
    wp_localize_script( 'business-one-page-pro-media-uploader', 'business_one_page_pro_uploader', array(
        'upload' => __( 'Upload', 'business-one-page-pro' ),
        'change' => __( 'Change', 'business-one-page-pro' ),
        'msg'    => __( 'Please upload valid image file.', 'business-one-page-pro' )
    ));
    
    wp_localize_script( 'business-one-page-pro-admin-js', 'business_one_page_pro_admin', array(
        'import_true' => __( 'Are you sure to import dummy content?', 'business-one-page-pro' ),
        'demo_msg'    => __( 'The Demo Contents are Loading. It might take a while. Please keep patience.', 'business-one-page-pro' ),
        'success_msg' => __( 'Demo Contents Successfully Imported.', 'business-one-page-pro' ),
    ));
    
    //For Color Picker
    wp_enqueue_script( 'business-one-page-pro-color-picker', get_template_directory_uri() . '/inc/js/color-picker.js', array( 'wp-color-picker' ), BUSINESS_ONE_PAGE_PRO_THEME_VERSION );    
}
endif;
add_action( 'admin_enqueue_scripts', 'business_one_page_pro_admin_scripts' );
add_action( 'elementor/editor/before_enqueue_scripts', 'business_one_page_pro_admin_scripts' );

if( ! function_exists( 'business_one_page_pro_customizer_js' ) ) :
/** 
 * Registering and enqueuing scripts/stylesheets for Customizer controls.
 */ 
function business_one_page_pro_customizer_js(){
    $array = array(
        'home'     => get_home_url(),
        'blog'     => business_one_page_pro_get_blog_page_url(),
        'about'    => business_one_page_pro_get_page_template_url( 'templates/template-about.php' ),
        'services' => business_one_page_pro_get_page_template_url( 'templates/template-services.php' ),
        'contact'  => business_one_page_pro_get_page_template_url( 'templates/template-contact.php' ),
    );
    wp_enqueue_script( 'business-one-page-pro-customizer-js', get_template_directory_uri() . '/inc/js/customizer.js', array( 'jquery', 'customize-controls' ), '20160523', true  );
    wp_localize_script( 'business-one-page-pro-customizer-js', 'bopp_cdata', $array );
}
endif;
add_action( 'customize_controls_enqueue_scripts', 'business_one_page_pro_customizer_js' );

if( ! function_exists( 'business_one_page_pro_body_classes' ) ) :
/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function business_one_page_pro_body_classes( $classes ) {
	
    $bg_color   = get_theme_mod( 'business_one_page_pro_bg_color', '#ffffff' );
    $bg_image   = get_theme_mod( 'business_one_page_pro_bg_image' );
    $bg_pattern = get_theme_mod( 'business_one_page_pro_bg_pattern', 'nobg' );
    $bg         = get_theme_mod( 'business_one_page_pro_body_bg', 'image' );
    $ed_section = get_theme_mod( 'business_one_page_pro_ed_section_menu' );
    
    // Adds a class of group-blog to blogs with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}
    
    // Adds a class for custom background Color
    if( $bg_color != '#ffffff' ){
        $classes[] = 'custom-background-color';
    }
    
    // Adds a class for custom background Color
    if( ( $bg == 'image' && $bg_image ) || (  $bg == 'pattern' && $bg_pattern != 'nobg' ) ){
        $classes[] = 'custom-background-image';
    }

    if( ! $ed_section ){
        $classes[] = 'section-menu';   
    }
    
    $classes[] = business_one_page_pro_sidebar( false, true );
                
	return $classes;
}
endif;
add_filter( 'body_class', 'business_one_page_pro_body_classes' );

if( ! function_exists( 'business_one_page_pro_post_classes' ) ) :
/**
 * Adds custom classes to the array of post classes.
 *
 * @param array $classes Classes for the post element.
 * @return array
 */
function business_one_page_pro_post_classes( $classes ){    
    $classes[] = 'latest_post';                             
    return $classes;
}
endif;
add_filter( 'post_class', 'business_one_page_pro_post_classes' );

/**
 * Custom CSS
*/
if ( function_exists( 'wp_update_custom_css_post' ) ) {
    // Migrate any existing theme CSS to the core option added in WordPress 4.7.
    $css = get_theme_mod( 'business_one_page_pro_custom_css' );
    if ( $css ) {
        $core_css = wp_get_custom_css(); // Preserve any CSS already added to the core option.
        $return = wp_update_custom_css_post( $core_css . $css );
        if ( ! is_wp_error( $return ) ) {
            // Remove the old theme_mod, so that the CSS is stored in only one place moving forward.
            remove_theme_mod( 'business_one_page_pro_custom_css' );
        }
    }
} else {
    // Back-compat for WordPress < 4.7.
    function business_one_page_pro_custom_css(){
        $custom_css = get_theme_mod( 'business_one_page_pro_custom_css' );
        if( !empty( $custom_css ) ){
    		echo '<style type="text/css">';
    		echo wp_strip_all_tags( $custom_css );
    		echo '</style>';
    	}
    }
    add_action( 'wp_head', 'business_one_page_pro_custom_css', 101 );
}

if( ! function_exists( 'business_one_page_pro_custom_js' ) ) :
/**
 * Custom JS
*/
function business_one_page_pro_custom_js(){
    $custom_script = get_theme_mod( 'business_one_page_pro_custom_script' );
    if( $custom_script ){
        echo '<script type="text/javascript">';
		echo wp_strip_all_tags( $custom_script );
		echo '</script>';
    }
}
endif;
add_action( 'wp_footer', 'business_one_page_pro_custom_js' );

if( ! function_exists( 'business_one_page_pro_category_transient_flusher' ) ) :
/**
 * Flush out the transients used in business_one_page_pro_categorized_blog.
 */
function business_one_page_pro_category_transient_flusher() {
	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}
	// Like, beat it. Dig?
	delete_transient( 'business_one_page_pro_categories' );
}
endif;
add_action( 'edit_category', 'business_one_page_pro_category_transient_flusher' );
add_action( 'save_post', 'business_one_page_pro_category_transient_flusher' );

if( ! function_exists( 'business_one_page_pro_logo_metabox' ) ) :
/**
 * Function that removes the default thumbnail and add custom thumbnail meta box 
 */
function business_one_page_pro_logo_metabox(){
   remove_meta_box( 'postimagediv', 'logo', 'side' );
   add_meta_box( 'postimagediv', __( 'Featured Image', 'business-one-page-pro' ), 'post_thumbnail_meta_box', 'logo', 'normal', 'high' );
}
endif;
add_action( 'do_meta_boxes', 'business_one_page_pro_logo_metabox' );

if( ! function_exists( 'business_one_page_pro_excerpts_in_pages' ) ) :
/**
 * Function to add excerpt field in pages
*/
function business_one_page_pro_excerpts_in_pages() {
     add_post_type_support( 'page', 'excerpt' );
}
endif;
add_action( 'init', 'business_one_page_pro_excerpts_in_pages' );

if( ! function_exists( 'business_one_page_pro_google_map_cb' ) ) :
/**
 * Callback Function for Google Map 
 * @link https://developers.google.com/maps/documentation/javascript/examples/marker-simple#try-it-yourself
*/
function business_one_page_pro_google_map_cb(){
    if( is_page_template( 'templates/template-contact.php' ) ){
        
        $ed_google_map     = get_theme_mod( 'business_one_page_pro_ed_google_map' );
        $google_map_option = get_theme_mod( 'business_one_page_pro_google_map_option', 'google_map_api' );
        $lattitude         = get_theme_mod( 'business_one_page_pro_lattitude', '27.7304135' );
        $longitude         = get_theme_mod( 'business_one_page_pro_longitude', '85.3304937' );
        $map_height        = get_theme_mod( 'business_one_page_pro_map_height', '320' );
        $map_zoom          = get_theme_mod( 'business_one_page_pro_map_zoom', '17' );
        $map_type          = get_theme_mod( 'business_one_page_pro_map_type', 'ROADMAP' );
        $map_scroll        = ( get_theme_mod( 'business_one_page_pro_ed_map_scroll', '1' ) == 1 ) ? 'true' : 'false';
        $map_control       = ( get_theme_mod( 'business_one_page_pro_ed_map_controls', '1' ) != 1 ) ? 'true' : 'false';
        $map_control_inv   = ( get_theme_mod( 'business_one_page_pro_ed_map_controls', '1' ) == 1 ) ? 'true' : 'false';
        $ed_map_marker     = get_theme_mod( 'business_one_page_pro_ed_map_marker' );
        $marker_title      = get_theme_mod( 'business_one_page_pro_marker_title' );
            
        if( $ed_google_map && 'google_map_api' == $google_map_option ){ ?>
        <style>
          #map {
            width: 100%;
            height: <?php echo $map_height; ?>px;
          }
        </style>
        <script type="text/javascript">
        
            var map;
            var myLatLng = {lat: <?php echo $lattitude;?>, lng: <?php echo $longitude;?>};
            
            function initialize() {
                var mapOptions = {
                    zoom: <?php echo $map_zoom; ?>,
                    center: new google.maps.LatLng(<?php echo $lattitude;?>, <?php echo $longitude;?> ),
                    mapTypeId: google.maps.MapTypeId.<?php echo $map_type; ?>,
                    scrollwheel: <?php echo $map_scroll ?>,
                    zoomControl: <?php echo $map_control_inv ?>,
                    zoomControlOptions: {
                        style: google.maps.ZoomControlStyle.SMALL
                    },
                    mapTypeControl: <?php echo $map_control_inv ?>,
                    mapTypeControlOptions: {
                        style: google.maps.MapTypeControlStyle.DROPDOWN_MENU
                    },
                    disableDefaultUI: <?php echo $map_control ?>,
                };
                map = new google.maps.Map(document.getElementById('map'), mapOptions);
                
                <?php if( $ed_map_marker ){ ?>
                var marker = new google.maps.Marker({
                  position: myLatLng,
                  map: map,
                  <?php if( $marker_title ) echo 'title: "' . esc_html( $marker_title ) . '"' ?>
                });
                <?php } ?>
            }
            google.maps.event.addDomListener(window, 'load', initialize); 
          
        </script>
    <?php }
    }    
}
endif;
add_action( 'wp_head', 'business_one_page_pro_google_map_cb' );

if( ! function_exists( 'business_one_page_pro_ajax_search' ) ) :
/**
 * AJAX Search results
 */ 
function business_one_page_pro_ajax_search(){
    $query = $_REQUEST['q']; // It goes through esc_sql() in WP_Query
    $search_query = new WP_Query( array( 's' => $query, 'posts_per_page' => 3, 'post_status' => 'publish' )); 
    $search_count = new WP_Query( array( 's' => $query, 'posts_per_page' => -1, 'post_status' => 'publish' ));
    $search_count = $search_count->post_count;
    if ( !empty( $query ) && $search_query->have_posts() ) : 
        
        echo '<ul class="ajax-search-results">';
        while ( $search_query->have_posts() ) : $search_query->the_post(); ?>
        <li><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></li>   
        <?php
        endwhile;
        echo '</ul>';
        
        echo '<div class="ajax-search-meta"><span class="results-count">'.$search_count.' '.__( 'Results', 'business-one-page-pro' ).'</span><a href="'.get_search_link( $query ).'" class="results-link">'.__('Show all results.', 'business-one-page-pro').'</a></div>';
    else:
        echo '<div class="no-results">'.__( 'No results found.', 'business-one-page-pro' ).'</div>';
    endif;
    
    wp_die(); // this is required to terminate immediately and return a proper response
}
endif;
if( get_theme_mod( 'business_one_page_pro_ed_ajax_search' ) ) {
    add_action( 'wp_ajax_business_one_page_pro_search', 'business_one_page_pro_ajax_search' );
    add_action( 'wp_ajax_nopriv_business_one_page_pro_search', 'business_one_page_pro_ajax_search' );
}

if( ! function_exists( 'business_one_page_pro_exclude_cat' ) ) :
/**
 * Exclude post with Category from blog and archive page. 
*/
function business_one_page_pro_exclude_cat( $query ){
    $cat = get_theme_mod( 'business_one_page_pro_exclude_categories' );    
    if( $cat && ! is_admin() && $query->is_main_query() ){
        $cat = array_diff( array_unique( $cat ), array('') );
        if( $query->is_home() || $query->is_archive() ) {
			$query->set( 'category__not_in', $cat );
		}
    }    
}
endif;
add_filter( 'pre_get_posts', 'business_one_page_pro_exclude_cat' );

if( ! function_exists( 'business_one_page_pro_custom_category_widget' ) ) :
/** 
 * Exclude Categories from Category Widget 
*/ 
function business_one_page_pro_custom_category_widget( $arg ) {
	$cat = get_theme_mod( 'business_one_page_pro_exclude_categories' );    
    if( $cat ){
        $cat = array_diff( array_unique( $cat ), array('') );
        $arg["exclude"] = $cat;
    }
	return $arg;
}
endif;
add_filter( "widget_categories_args", "business_one_page_pro_custom_category_widget" );
add_filter( "widget_categories_dropdown_args", "business_one_page_pro_custom_category_widget" );

if( ! function_exists( 'business_one_page_pro_exclude_posts_from_recentPostWidget_by_cat' ) ) :
/**
 * Exclude post from recent post widget of excluded catergory
 * 
 * @link http://blog.grokdd.com/exclude-recent-posts-by-category/
*/
function business_one_page_pro_exclude_posts_from_recentPostWidget_by_cat( $arg ){    
    $cat = get_theme_mod( 'business_one_page_pro_exclude_categories' );
    if( $cat ){
        $cat = array_diff( array_unique( $cat ), array('') );
        $arg["category__not_in"] = $cat;
    }    
    return $arg;   
}
endif;
add_filter( "widget_posts_args", "business_one_page_pro_exclude_posts_from_recentPostWidget_by_cat" );

if( ! function_exists( 'business_one_page_pro_allowed_social_protocols' ) ) :
/**
 * List of allowed social protocols in HTML attributes.
 * @param  array $protocols Array of allowed protocols.
 * @return array
 */
function business_one_page_pro_allowed_social_protocols( $protocols ) {
    $social_protocols = array(
        'skype'
    );
    return array_merge( $protocols, $social_protocols );    
}
endif;
add_filter( 'kses_allowed_protocols' , 'business_one_page_pro_allowed_social_protocols' );

if( ! function_exists( 'business_one_page_pro_theme_updater' ) ) :
/**
 * Load theme updater functions.
 * Action is used so that child themes can easily disable.
 */
function business_one_page_pro_theme_updater() {
    require( get_template_directory() . '/updater/theme-updater.php' );
}
endif;
add_action( 'after_setup_theme', 'business_one_page_pro_theme_updater' );

if( ! function_exists( 'business_one_page_pro_single_post_schema' ) ) :
/**
 * Single Post Schema
 *
 * @return string
 */
function business_one_page_pro_single_post_schema() {
    if ( is_singular( 'post' ) ) {
        global $post;
        $custom_logo_id = get_theme_mod( 'custom_logo' );

        $site_logo   = wp_get_attachment_image_src( $custom_logo_id , 'business-one-page-pro-schema' );
        $images      = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'full' );
        $excerpt     = business_one_page_pro_escape_text_tags( $post->post_excerpt );
        $content     = $excerpt === "" ? mb_substr( business_one_page_pro_escape_text_tags( $post->post_content ), 0, 110 ) : $excerpt;
        $schema_type = ! empty( $custom_logo_id ) && has_post_thumbnail( $post->ID ) ? "BlogPosting" : "Blog";

        $args = array(
            "@context"  => "http://schema.org",
            "@type"     => $schema_type,
            "mainEntityOfPage" => array(
                "@type" => "WebPage",
                "@id"   => get_permalink( $post->ID )
            ),
            "headline"  => ( function_exists( '_wp_render_title_tag' ) ? wp_get_document_title() : wp_title( '', false, 'right' ) ),
            "image"     => array(
                "@type"  => "ImageObject",
                "url"    => $images[0],
                "width"  => $images[1],
                "height" => $images[2]
            ),
            "datePublished" => get_the_time( DATE_ISO8601, $post->ID ),
            "dateModified"  => get_post_modified_time(  DATE_ISO8601, __return_false(), $post->ID ),
            "author"        => array(
                "@type"     => "Person",
                "name"      => business_one_page_pro_escape_text_tags( get_the_author_meta( 'display_name', $post->post_author ) )
            ),
            "publisher" => array(
                "@type"       => "Organization",
                "name"        => get_bloginfo( 'name' ),
                "description" => get_bloginfo( 'description' ),
                "logo"        => array(
                    "@type"   => "ImageObject",
                    "url"     => $site_logo[0],
                    "width"   => $site_logo[1],
                    "height"  => $site_logo[2]
                )
            ),
            "description" => ( class_exists('WPSEO_Meta') ? WPSEO_Meta::get_value( 'metadesc' ) : $content )
        );

        if ( has_post_thumbnail( $post->ID ) ) :
            $args['image'] = array(
                "@type"  => "ImageObject",
                "url"    => $images[0],
                "width"  => $images[1],
                "height" => $images[2]
            );
        endif;

        if ( ! empty( $custom_logo_id ) ) :
            $args['publisher'] = array(
                "@type"       => "Organization",
                "name"        => get_bloginfo( 'name' ),
                "description" => get_bloginfo( 'description' ),
                "logo"        => array(
                    "@type"   => "ImageObject",
                    "url"     => $site_logo[0],
                    "width"   => $site_logo[1],
                    "height"  => $site_logo[2]
                )
            );
        endif;

        echo '<script type="application/ld+json">' , PHP_EOL;
        echo wp_json_encode( $args, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT ) , PHP_EOL;
        echo '</script>' , PHP_EOL;
    }
}
endif;
add_action( 'wp_head', 'business_one_page_pro_single_post_schema' );

if( ! function_exists( 'business_one_page_pro_filter_post_gallery' ) ) :
/**
 * Filter the output of the gallery. 
*/ 
function business_one_page_pro_filter_post_gallery( $output, $attr, $instance ){
    global $post, $wp_locale;

    $html5 = current_theme_supports( 'html5', 'gallery' );
    $atts = shortcode_atts( array(
        'order'      => 'ASC',
        'orderby'    => 'menu_order ID',
        'id'         => $post ? $post->ID : 0,
        'itemtag'    => $html5 ? 'figure'     : 'dl',
        'icontag'    => $html5 ? 'div'        : 'dt',
        'captiontag' => $html5 ? 'figcaption' : 'dd',
        'columns'    => 3,
        'size'       => 'thumbnail',
        'include'    => '',
        'exclude'    => '',
        'link'       => ''
    ), $attr, 'gallery' );
    
    $id = intval( $atts['id'] );
    
    if ( ! empty( $atts['include'] ) ) {
        $_attachments = get_posts( array( 'include' => $atts['include'], 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $atts['order'], 'orderby' => $atts['orderby'] ) );
    
        $attachments = array();
        foreach ( $_attachments as $key => $val ) {
            $attachments[$val->ID] = $_attachments[$key];
        }
    } elseif ( ! empty( $atts['exclude'] ) ) {
        $attachments = get_children( array( 'post_parent' => $id, 'exclude' => $atts['exclude'], 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $atts['order'], 'orderby' => $atts['orderby'] ) );
    } else {
        $attachments = get_children( array( 'post_parent' => $id, 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $atts['order'], 'orderby' => $atts['orderby'] ) );
    }
    
    if ( empty( $attachments ) ) {
        return '';
    }
    
    if ( is_feed() ) {
        $output = "\n";
        foreach ( $attachments as $att_id => $attachment ) {
            $output .= wp_get_attachment_link( $att_id, $atts['size'], true ) . "\n";
        }
        return $output;
    }
    
    $itemtag = tag_escape( $atts['itemtag'] );
    $captiontag = tag_escape( $atts['captiontag'] );
    $icontag = tag_escape( $atts['icontag'] );
    $valid_tags = wp_kses_allowed_html( 'post' );
    if ( ! isset( $valid_tags[ $itemtag ] ) ) {
        $itemtag = 'dl';
    }
    if ( ! isset( $valid_tags[ $captiontag ] ) ) {
        $captiontag = 'dd';
    }
    if ( ! isset( $valid_tags[ $icontag ] ) ) {
        $icontag = 'dt';
    }
    
    $columns = intval( $atts['columns'] );
    $itemwidth = $columns > 0 ? floor(100/$columns) : 100;
    $float = is_rtl() ? 'right' : 'left';
    
    $selector = "gallery-{$instance}";
    
    $gallery_style = '';
    
    /**
     * Filter whether to print default gallery styles.
     *
     * @since 3.1.0
     *
     * @param bool $print Whether to print default gallery styles.
     *                    Defaults to false if the theme supports HTML5 galleries.
     *                    Otherwise, defaults to true.
     */
    if ( apply_filters( 'use_default_gallery_style', ! $html5 ) ) {
        $gallery_style = "
        <style type='text/css'>
            #{$selector} {
                margin: auto;
            }
            #{$selector} .gallery-item {
                float: {$float};
                margin-top: 10px;
                text-align: center;
                width: {$itemwidth}%;
            }
            #{$selector} img {
                border: 2px solid #cfcfcf;
            }
            #{$selector} .gallery-caption {
                margin-left: 0;
            }
            /* see gallery_shortcode() in wp-includes/media.php */
        </style>\n\t\t";
    }
    
    $size_class = sanitize_html_class( $atts['size'] );
    $gallery_div = "<div id='$selector' class='gallery galleryid-{$id} gallery-columns-{$columns} gallery-size-{$size_class}'>";
    
    /**
     * Filter the default gallery shortcode CSS styles.
     *
     * @since 2.5.0
     *
     * @param string $gallery_style Default CSS styles and opening HTML div container
     *                              for the gallery shortcode output.
     */
    $output = apply_filters( 'gallery_style', $gallery_style . $gallery_div );
    
    $i = 0; 
    foreach ( $attachments as $id => $attachment ) {
            
        $attr = ( trim( $attachment->post_excerpt ) ) ? array( 'aria-describedby' => "$selector-$id" ) : '';
        if ( ! empty( $atts['link'] ) && 'file' === $atts['link'] ) {
            //$image_output = wp_get_attachment_link( $id, $atts['size'], false, false, false, $attr ); // for attachment url 
            $image_output = "<a href='" . wp_get_attachment_url( $id ) . "' data-fancybox='group{$columns}' data-caption='" . esc_attr( $attachment->post_excerpt ) . "'>";
            $image_output .= wp_get_attachment_image( $id, $atts['size'], false, $attr );
            $image_output .= "</a>";
        } elseif ( ! empty( $atts['link'] ) && 'none' === $atts['link'] ) {
            $image_output = wp_get_attachment_image( $id, $atts['size'], false, $attr );
        } else {
            $image_output = wp_get_attachment_link( $id, $atts['size'], true, false, false, $attr ); //for attachment page
        }
        $image_meta  = wp_get_attachment_metadata( $id );
    
        $orientation = '';
        if ( isset( $image_meta['height'], $image_meta['width'] ) ) {
            $orientation = ( $image_meta['height'] > $image_meta['width'] ) ? 'portrait' : 'landscape';
        }
        $output .= "<{$itemtag} class='gallery-item'>";
        $output .= "
            <{$icontag} class='gallery-icon {$orientation}'>
                $image_output
            </{$icontag}>";
        if ( $captiontag && trim($attachment->post_excerpt) ) {
            $output .= "
                <{$captiontag} class='wp-caption-text gallery-caption' id='$selector-$id'>
                " . wptexturize($attachment->post_excerpt) . "
                </{$captiontag}>";
        }
        $output .= "</{$itemtag}>";
        if ( ! $html5 && $columns > 0 && ++$i % $columns == 0 ) {
            $output .= '<br style="clear: both" />';
        }
    }
    
    if ( ! $html5 && $columns > 0 && $i % $columns !== 0 ) {
        $output .= "
            <br style='clear: both' />";
    }
    
    $output .= "
        </div>\n";
    
    return $output;
}
endif;

if ( class_exists( 'Jetpack' ) ){
    if( ! Jetpack::is_module_active( 'carousel' ) ){
        add_filter( 'post_gallery', 'business_one_page_pro_filter_post_gallery', 10, 3 );
    }
}else{
    add_filter( 'post_gallery', 'business_one_page_pro_filter_post_gallery', 10, 3 );
}

if( ! function_exists( 'business_one_page_pro_migrate_free_options' ) ) :
/**
 * Function to migrate free theme option to pro theme
*/
function business_one_page_pro_migrate_free_options(){
    $fresh              = get_option( '_business_one_page_pro_fresh_install' ); //flag to check if it is first switch
    $blank_about_post   = array();
    $blank_service_post = array();
    $icon_text          = array();
    $about_sidebar      = array();
    $service_sidebar    = array();
    $cta_one            = array();
    $cta_two            = array();
    $cta_top            = array();
    $cta_bot            = array();
    $cta                = array();
    $social_array       = array();
    $final_about        = array();
    $final_service      = array();
    $final_cta_top      = array();
    $final_cta_bottom   = array();

    $pages = array(
        'about-us' => array( 
            'page_name'     => esc_html__( 'About Us', 'business-one-page-pro' ),
            'page_template' => 'templates/template-about.php'
        ),
        'contact-us' => array( 
            'page_name'     => esc_html__( 'Contact Us', 'business-one-page-pro' ),
            'page_template' => 'templates/template-contact.php'
        ),
        'gallery-page' => array( 
            'page_name'     => esc_html__( 'Gallery Page', 'business-one-page-pro' ),
            'page_template' => 'templates/template-gallery.php'
        ),
        'homepage' => array( 
            'page_name'     => esc_html__( 'Homepage', 'business-one-page-pro' ),
            'page_template' => 'templates/template-home.php'
        ),
        'our-portfolio' => array( 
            'page_name'     => esc_html__( 'Our Portfolio', 'business-one-page-pro' ),
            'page_template' => 'templates/template-portfolio.php'
        ),
        'services' => array( 
            'page_name'     => esc_html__( 'Services', 'business-one-page-pro' ),
            'page_template' => 'templates/template-services.php'
        ),
        'our-team' => array( 
            'page_name'     => esc_html__( 'Our Team', 'business-one-page-pro' ),
            'page_template' => 'templates/template-team.php'
        ),
        'testimonials' => array( 
            'page_name'     => esc_html__( 'Testimonials', 'business-one-page-pro' ),
            'page_template' => 'templates/template-testimonial.php'
        ),
    );
    
    $i = 1;
    $j = 1;
    
    if( ! $fresh ){        
        $options         = get_option( 'theme_mods_business-one-page' );
        $sidebar_options = get_option( 'sidebars_widgets' );
        
        if( $options ){
            //migrate free theme option to pro
            foreach( $options as $option => $value ){
                if( $option == 'business_one_page_ed_home_link' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_ed_home_link', true );
                }elseif( $option == 'business_one_page_ed_secion_menu' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_ed_section_menu', true );                    
                }elseif( $option == 'business_one_page_home_link_label' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_home_link_label', $value );
                }elseif( $option == 'business_one_page_ed_slider' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_ed_slider', true );
                }elseif( $option == 'business_one_page_slider_auto' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_auto', true );
                }elseif( $option == 'business_one_page_slider_loop' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_loop', true );
                }elseif( $option == 'business_one_page_slider_pager' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_pager', true );
                }elseif( $option == 'business_one_page_slider_caption' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_caption', true );
                }elseif( $option == 'business_one_page_slider_animation' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_animation', $value );
                }elseif( $option == 'business_one_page_slider_speeds' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_speed', $value );
                }elseif( $option == 'business_one_page_slider_pause' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_pause', $value );
                }elseif( $option == 'business_one_page_slider_readmore' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_readmore', $value );
                }elseif( $option == 'business_one_page_slider_post_one' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_post_one', $value );
                }elseif( $option == 'business_one_page_slider_post_two' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_post_two', $value );
                }elseif( $option == 'business_one_page_slider_post_three' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_post_three', $value );
                }elseif( $option == 'business_one_page_slider_post_four' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_slider_post_four', $value );
                }elseif( $option == 'business_one_page_about_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_about_section_menu_title', $value );
                }elseif( $option == 'business_one_page_about_section_page' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_about_section_page', $value );
                }elseif( $option == 'business_one_page_about_section_post_one' ){
                    if( $value ) array_push( $blank_about_post, $value );
                }elseif( $option == 'business_one_page_about_section_post_two' ){
                    if( $value ) array_push( $blank_about_post, $value );
                }elseif( $option == 'business_one_page_about_section_post_three' ){
                    if( $value ) array_push( $blank_about_post, $value );
                }elseif( $option == 'business_one_page_services_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_services_section_menu_title', $value );
                }elseif( $option == 'business_one_page_services_section_page' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_services_section_page', $value );
                }elseif( $option == 'business_one_page_services_section_post_one' ){
                    if( $value ) array_push( $blank_service_post, $value );
                }elseif( $option == 'business_one_page_services_section_post_two' ){
                    if( $value ) array_push( $blank_service_post, $value );
                }elseif( $option == 'business_one_page_services_section_post_three' ){
                    if( $value ) array_push( $blank_service_post, $value );
                }elseif( $option == 'business_one_page_services_section_post_four' ){
                    if( $value ) array_push( $blank_service_post, $value );
                }elseif( $option == 'business_one_page_services_section_post_five' ){
                    if( $value ) array_push( $blank_service_post, $value );
                }elseif( $option == 'business_one_page_services_section_post_six' ){
                    if( $value ) array_push( $blank_service_post, $value );
                }elseif(  $option == 'business_one_page_ed_cta1_section' || $option == 'business_one_page_cta1_section_title' || $option == 'business_one_page_cta1_section_content' || $option == 'business_one_page_cta1_section_button' || $option == 'business_one_page_cta1_section_button_url' ){                
                    if( $option == 'business_one_page_cta1_section_title' ) $cta_one['title'] = $value; 
                    if( $option == 'business_one_page_cta1_section_content' ) $cta_one['content'] = $value;
                    if( $option == 'business_one_page_cta1_section_button' ) $cta_one['button_text'] = $value;
                    if( $option == 'business_one_page_cta1_section_button_url' ) $cta_one['button_url'] = $value;
                    if( $option == 'business_one_page_ed_cta1_section' ){
                        if( $value ){
                            $cta_one['style'] = 'style1';
                            $cta_one['bg_color'] = '#32c4d1';
                            $cta_one['bg_image'] = '';        
                        }
                    }                
                }elseif( $option == 'business_one_page_portfolio_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_portfolio_section_menu_title', $value );
                }elseif( $option == 'business_one_page_portfolio_section_page' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_portfolio_section_page', $value );
                }elseif( $option == 'business_one_page_portfolio_section_post_one' ){
                    $post_id = business_one_page_pro_duplicate_post( $value, 'portfolio' );
                    if( $post_id ) set_theme_mod( 'business_one_page_pro_portfolio_section_post_one', $post_id );
                }elseif( $option == 'business_one_page_portfolio_section_post_two' ){
                    $post_id = business_one_page_pro_duplicate_post( $value, 'portfolio' );
                    if( $post_id ) set_theme_mod( 'business_one_page_pro_portfolio_section_post_two', $post_id );
                }elseif( $option == 'business_one_page_portfolio_section_post_three' ){
                    $post_id = business_one_page_pro_duplicate_post( $value, 'portfolio' );
                    if( $post_id ) set_theme_mod( 'business_one_page_pro_portfolio_section_post_three', $post_id );
                }elseif( $option == 'business_one_page_portfolio_section_post_four' ){
                    $post_id = business_one_page_pro_duplicate_post( $value, 'portfolio' );
                    if( $post_id ) set_theme_mod( 'business_one_page_pro_portfolio_section_post_four', $post_id );
                }elseif( $option == 'business_one_page_portfolio_section_post_five' ){
                    $post_id = business_one_page_pro_duplicate_post( $value, 'portfolio' );
                    if( $post_id ) set_theme_mod( 'business_one_page_pro_portfolio_section_post_five', $post_id );
                }elseif( $option == 'business_one_page_team_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_team_section_menu_title', $value );
                }elseif( $option == 'business_one_page_team_section_page' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_team_section_page', $value );
                }elseif( $option == 'business_one_page_team_section_cat' ){
                    if( $value ){
                        $posts = get_posts( array( 'numberposts' => -1, 'category' => $value ) );
                        if( $posts ){
                            foreach( $posts as $p ){
                                business_one_page_pro_duplicate_post( $p->ID, 'team' );
                            }
                        }
                    }
                }elseif( $option == 'business_one_page_clients_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_clients_section_menu_title', $value );
                }elseif( $option == 'business_one_page_clients_section_page' ){
                    if( $value ){
                        $title = get_the_title( $value );
                        set_theme_mod( 'business_one_page_pro_clients_section_title', $title );                        
                    }
                }elseif( $option == 'business_one_page_clients_section_logos' ){
                    if( $value ){
                        $logos = explode( ',', $value );
                        foreach( $logos as $logo ){
                            $logo_id = business_one_page_pro_get_attachment_id( $logo );
                            if( $logo_id ){
                                business_one_page_pro_duplicate_post( $logo_id, 'logo', true );                                
                            }
                        }
                    }
                }elseif( $option == 'business_one_page_blog_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_blog_section_menu_title', $value );
                }elseif( $option == 'business_one_page_blog_section_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_blog_section_title', $value );
                }elseif( $option == 'business_one_page_blog_section_content' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_blog_section_content', $value );
                }elseif( $option == 'business_one_page_blog_section_view_all' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_blog_section_view_all', $value );
                }elseif( $option == 'business_one_page_testimonial_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_testimonial_section_menu_title', $value );
                }elseif( $option == 'business_one_page_testimonial_section_page' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_testimonial_section_page', $value );
                }elseif( $option == 'business_one_page_testimonial_section_cat' ){
                    if( $value ){
                        $posts = get_posts( array( 'numberposts' => -1, 'category' => $value ) );
                        if( $posts ){
                            foreach( $posts as $p ){
                                business_one_page_pro_duplicate_post( $p->ID, 'testimonial' );
                            }
                        }
                    }
                }elseif( $option == 'business_one_page_ed_cta2_section' || $option == 'business_one_page_cta2_section_title' || $option == 'business_one_page_cta2_section_content' || $option == 'business_one_page_cta2_section_button' || $option == 'business_one_page_cta2_section_button_url' ){                
                    if( $option == 'business_one_page_cta2_section_title' ) $cta_two['title'] = $value;
                    if( $option == 'business_one_page_cta2_section_content' ) $cta_two['content'] = $value;
                    if( $option == 'business_one_page_cta2_section_button' ) $cta_two['button_text'] = $value;
                    if( $option == 'business_one_page_cta2_section_button_url' ) $cta_two['button_url'] = $value;
                    if( $option == 'business_one_page_ed_cta2_section' ){
                        if( $value ){
                            $cta_two['style'] = 'style1';
                            $cta_two['bg_color'] = '#32c4d1';
                            $cta_two['bg_image'] = '';        
                        }
                    }                                    
                }elseif( $option == 'business_one_page_contact_section_menu_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_menu_title', $value );
                }elseif( $option == 'business_one_page_contact_section_page' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_page', $value );
                }elseif( $option == 'business_one_page_contact_form_label' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_form_title', $value );
                }elseif( $option == 'business_one_page_contact_section_form' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_form', $value );
                }elseif( $option == 'business_one_page_contact_section_info_title' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_info_title', $value );
                }elseif( $option == 'business_one_page_contact_section_info_content' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_info_content', $value );
                }elseif( $option == 'business_one_page_contact_section_address' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_address', $value );
                }elseif( $option == 'business_one_page_contact_section_phone' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_phone', $value );
                }elseif( $option == 'business_one_page_contact_section_fax' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_fax', $value );
                }elseif( $option == 'business_one_page_contact_section_email' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_contact_section_email', $value );
                }elseif( $option == 'business_one_page_facebook' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'facebook', 'link' => $value ) );
                }elseif( $option == 'business_one_page_twitter' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'twitter', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_pinterest' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'pinterest', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_linkedin' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'linkedin', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_instagram' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'instagram', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_youtube' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'youtube', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_odnoklassniki' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'odnoklassniki', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_vk' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'vk', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_xing' ){
                    if( $value ) array_push( $social_array, array( 'icon' => 'xing', 'link'  => $value ) );
                }elseif( $option == 'business_one_page_exclude_cat' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_exclude_categories', $value );
                }elseif( $option == 'business_one_page_footer_copyright_text' ){
                    if( $value ) set_theme_mod( 'business_one_page_pro_footer_copyright', $value );
                }elseif( $option !== 'sidebars_widgets' ){
                    set_theme_mod( $option, $value );
                }    
            }
            
            if( $social_array ){
                set_theme_mod( 'business_one_page_pro_social', $social_array );
            }
            
            //Let's create sidebar progamatically            
            if( $blank_about_post ){
                foreach( $blank_about_post as $about ){
                    $p = get_post( $about );
                    $icon_text[$i] = array(
                        'title'   => $p->post_title,
                        'content' => $p->post_content,
                        'style'   => 'style1',
                        'image'   => get_post_thumbnail_id( $p->ID ),
                        'icon'    => '',
                        'url'     => '',
                        'new_tab' => '',
                    );
                    array_push( $about_sidebar, 'business_one_page_pro_icon_text_widget-' . $i );
                    $i++;                                        
                }
            }
            
            if( $blank_service_post ){
                foreach( $blank_service_post as $service ){
                    $s = get_post( $service );
                    $icon_text[$i] = array(
                        'title'   => $s->post_title,
                        'content' => $s->post_content,
                        'style'   => 'style2',
                        'image'   => get_post_thumbnail_id( $s->ID ),
                        'icon'    => '',
                        'url'     => '',
                        'new_tab' => '',
                    );
                    array_push( $service_sidebar, 'business_one_page_pro_icon_text_widget-' . $i );
                    $i++;
                }
            }
            
            if( $icon_text ){
                update_option( 'widget_business_one_page_pro_icon_text_widget', $icon_text );
            }
            
            if( $cta_one ){
                $cta[$j] = $cta_one;
                array_push( $cta_top, 'business_one_page_pro_cta_widget-' . $j );               
                $j++;
            } 
            if( $cta_two ){
                $cta[$j] = $cta_two;
                array_push( $cta_bot, 'business_one_page_pro_cta_widget-' . $j );
            }
            
            if( $cta ){
                update_option( 'widget_business_one_page_pro_cta_widget', $cta );
            }
                            
            if( $about_sidebar ){
                $final_about = array( 'about-section' => $about_sidebar );
            }
            if( $service_sidebar ){
                $final_service = array( 'service-section' => $service_sidebar );
            }
            if( $cta_top ){
                $final_cta_top = array( 'cta-one' => $cta_top );
            }
            if( $cta_bot ){
                $final_cta_bottom =  array( 'cta-two' => $cta_bot );
            }
            
            $final_sidebar = array_merge( $sidebar_options, $final_about, $final_service, $final_cta_top, $final_cta_bottom );
            update_option( 'sidebars_widgets', $final_sidebar );            
        
            //create default pages on theme migration
            foreach( $pages as $page => $val ){
                business_one_page_pro_create_post( $val['page_name'], $page, $val['page_template'] );
            }
        }
        update_option( '_business_one_page_pro_fresh_install', true );  
    }
}
endif;
add_action( 'after_switch_theme', 'business_one_page_pro_migrate_free_options' );

if( ! function_exists( 'business_one_page_pro_update_fresh_install' ) ) :
/**
 * This function updates the fresh install update for already installed pro theme.
 * This function will prevent accidental migration of free theme option to pro if
 * a old user has free theme installed and switching to other theme and again back 
 * to pro theme.   
*/
function business_one_page_pro_update_fresh_install(){
    update_option( '_business_one_page_pro_fresh_install', true );
}
endif;
add_action( 'switch_theme', 'business_one_page_pro_update_fresh_install' );

if( ! function_exists( 'business_one_page_pro_activate_notice' ) ) :
/**
 * Theme activation notice.
*/
function business_one_page_pro_activate_notice() {
    global $current_user;
    $theme_list       = wp_get_theme( 'business-one-page-pro' );
    $add_license      = get_option( 'business-one-page-pro_license_key' );
    $current_screen   = get_current_screen();    
    $activate_license = get_option( 'business-one-page-pro_license_key_status' );
    $statuses         = array( 'invalid', 'inactive', 'expired', 'disabled', 'site_inactive' );   
    
    if( $theme_list->exists() && ( empty( $add_license ) || in_array( $activate_license, $statuses ) ) && $current_screen->base != 'appearance_page_business-one-page-pro-license' ) { ?>
        <div class="notice notice-info is-dismissible">
            <p><?php esc_html_e( 'Activate Theme License ( Business One Page Pro ) to enjoy the full benefits of using the theme. We\'re sorry about this extra step but we built the activation to prevent mass piracy of our themes. This allows us to better serve our paying customers. An active theme comes with free updates, top notch support and guaranteed latest WordPress support.', 'business-one-page-pro' ); ?>
            </p>
            <p>
                <span style="color:red;"><?php esc_html_e( 'Please Activate Theme License!', 'business-one-page-pro' ); ?></span> - <a href="<?php echo esc_url( 'themes.php?page=business-one-page-pro-license' ); ?>"><u><?php esc_html_e( 'Click here to enter your license key', 'business-one-page-pro' ); ?></u></a> - <?php esc_html_e( 'if there is an error, please contact us at ', 'business-one-page-pro' ); ?><a href="mailto:support@raratheme.com" target="_blank">support@raratheme.com</a> - <a href="<?php echo esc_url( 'https://rarathemes.com/activate-theme-license/' ); ?>" target="_blank"><u><?php esc_html_e( 'How to activate the theme license', 'business-one-page-pro' ); ?></u></a>.
            </p> 
        </div>  
        <?php
    }
}
endif;
add_action( 'admin_notices', 'business_one_page_pro_activate_notice' );

if( ! function_exists( 'business_one_page_pro_redirect_on_activation' ) ) :
/**
 * Redirect to Getting Started page on theme activation
*/
function business_one_page_pro_redirect_on_activation() {
    global $pagenow;
    if ( is_admin() && 'themes.php' == $pagenow && isset( $_GET['activated'] ) ) {
        wp_redirect( admin_url( "themes.php?page=business-one-page-pro-license" ) );
    }
}
endif;
add_action( 'admin_init', 'business_one_page_pro_redirect_on_activation' );

/**
 * Remove issues with prefetching adding extra views 
*/
remove_action( 'wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0);

/**
 * Enable shortcodes in text widgets 
*/
add_filter( 'widget_text', 'do_shortcode' );