<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Business_One_Page_Pro
 */

if ( ! function_exists( 'business_one_page_pro_posted_on' ) ) :
/**
 * Prints HTML with meta information for the current post-date/time and author.
 */
function business_one_page_pro_posted_on( $inner = false ) {
	
    $post_meta_blog_page    = get_theme_mod( 'business_one_page_pro_post_meta_blog_page', array( 'date', 'author' ) );
    $post_meta_blog_section = get_theme_mod( 'business_one_page_pro_post_meta_blog_section', array( 'date', 'comment_count' ) );
    $i = 0;
    if( $inner && $post_meta_blog_page ){
        foreach( $post_meta_blog_page as $meta ){
            business_one_page_pro_meta( $meta );  
        }
    }elseif( $post_meta_blog_section ){
        foreach( $post_meta_blog_section as $meta ){
            $i++;
            business_one_page_pro_meta( $meta );
            if( $i < count($post_meta_blog_section) ){
                esc_html_e( '/', 'business-one-page-pro' );
            }  
        }
    }
    
    
}
endif;

if( ! function_exists( 'business_one_page_pro_meta' ) ) :
/**
 * List meta information
*/
function business_one_page_pro_meta( $meta ){
    
    switch( $meta ){
        
        case 'date':
        
        $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
	
    	$time_string = sprintf( $time_string, esc_attr( get_the_date( 'c' ) ), esc_html( get_the_date() ) );

        $posted_on = sprintf( esc_html_x( '%s', 'post date', 'business-one-page-pro' ), '<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a>' );
        
        echo '<span class="posted-on">' . $posted_on . '</span>';
        
        break;
        
        case 'author':
        
        $byline = sprintf(
            esc_html_x( '%s', 'post author', 'business-one-page-pro' ),
            '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span>'
            );
            
        echo '<span class="byline"> ' . $byline . '</span>';
        
        break;
        
        case 'comment_count':
        
        $comment_count = sprintf( _nx( '1 Comment', '%1$s Comments', get_comments_number(), 'comments title', 'business-one-page-pro' ), number_format_i18n( get_comments_number() ) ); 
	
        $comment = sprintf(
    		esc_html_x( '%s', 'post comment', 'business-one-page-pro' ),
    		'<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $comment_count . '</a>'
    	);
        echo '<span class="comments-link">' . $comment . '</span>';
        
        break;
        
    }
}
endif;

if ( ! function_exists( 'business_one_page_pro_entry_footer' ) ) :
/**
 * Prints edit links
 */
function business_one_page_pro_entry_footer() {	
	edit_post_link(
		sprintf(
			/* translators: %s: Name of current post */
			esc_html__( 'Edit %s', 'business-one-page-pro' ),
			the_title( '<span class="screen-reader-text">"', '"</span>', false )
		),
		'<span class="edit-link">',
		'</span>'
	);
}
endif;

if( ! function_exists( 'business_one_page_pro_cat_tag' ) ) :
/**
 * Categories and Tags
*/
function business_one_page_pro_cat_tag(){
    
    $cat_tag = get_theme_mod( 'business_one_page_pro_cat_tag', array( 'cat', 'tag' ) );
    
    // Hide category and tag text for pages.
	if ( 'post' === get_post_type() && $cat_tag ) {
		echo '<div class="tags-block">';
            foreach( $cat_tag as $c ){
                business_one_page_pro_tax( $c );
            }            
        echo '</div>';		
	}
}
endif;

if( ! function_exists( 'business_one_page_pro_tax' ) ) :
/**
 * List Cat and Tags
*/
function business_one_page_pro_tax( $tax ){
    switch( $tax ){
        
        case 'cat':
        
        /* translators: used between list items, there is a space after the comma */
		$categories_list = get_the_category_list( esc_html__( ', ', 'business-one-page-pro' ) );		
        
        if ( $categories_list && business_one_page_pro_categorized_blog() ) {
			printf( '<span class="cat-links"><span class="fa fa-folder-open"></span>' . esc_html__( 'Categories: %1$s', 'business-one-page-pro' ) . '</span>', $categories_list ); // WPCS: XSS OK.
		}
        
        break;
        
        case 'tag':
        
        /* translators: used between list items, there is a space after the comma */
		$tags_list = get_the_tag_list( '', esc_html__( ', ', 'business-one-page-pro' ) );
		if ( $tags_list ) {
			printf( '<span class="tags-links"><span class="fa fa-tags"></span>' . esc_html__( 'Tags: %1$s', 'business-one-page-pro' ) . '</span>', $tags_list ); // WPCS: XSS OK.
		}
        
        break;
    }    
}
endif;

if( ! function_exists( 'business_one_page_pro_categories' ) ) :
/**
 * Function that list categories
*/
function business_one_page_pro_categories(){
    $categories_list = get_the_category_list( esc_html__( ' ', 'business-one-page-pro' ) ); 
    if ( $categories_list && business_one_page_pro_categorized_blog() ) {
        printf( '<span class="category">' . esc_html__( '%1$s', 'business-one-page-pro' ) . '</span>', $categories_list ); // WPCS: XSS OK.
    }
}
endif;

if( ! function_exists( 'business_one_page_pro_portfolio_categories' ) ) :
/**
 * Portfolio Categories lists
*/
function business_one_page_pro_portfolio_categories(){
    global $post;
    $cat_list = get_the_term_list( $post->ID, 'portfolio-category', '', esc_html__( ', ', 'business-one-page-pro' ) );
    if( $cat_list ){
        printf( '<span class="category">' . esc_html__( '%1$s', 'business-one-page-pro' ) . '</span>', $cat_list ); // WPCS: XSS OK.
    }
}
endif;

if( ! function_exists( 'business_one_page_pro_author_details' ) ) :
/**
 * Author Bio
*/
function business_one_page_pro_author_details(){
    ?>
    <div class="author-block">
		<h2 class="author-title"><?php esc_html_e( 'About the Author', 'business-one-page-pro' ); ?></h2>
		<div class="author-holder">
			<div class="img-holder">
                <?php echo get_avatar( get_the_author_meta( 'ID' ), 130 ); ?>               
            </div>
			<div class="text-holder">
				<h3 class="author-name"><?php echo esc_html( get_the_author_meta( 'display_name' ) ); ?></h3>
				<?php echo wpautop( esc_html( get_the_author_meta( 'description' ) ) ); ?>
			</div>
		</div>
	</div>
    <?php
}
endif;

/**
 * Returns true if a blog has more than 1 category.
 *
 * @return bool
 */
function business_one_page_pro_categorized_blog() {
	if ( false === ( $all_the_cool_cats = get_transient( 'business_one_page_pro_categories' ) ) ) {
		// Create an array of all the categories that are attached to posts.
		$all_the_cool_cats = get_categories( array(
			'fields'     => 'ids',
			'hide_empty' => 1,
			// We only need to know if there is more than one category.
			'number'     => 2,
		) );

		// Count the number of categories that are attached to the posts.
		$all_the_cool_cats = count( $all_the_cool_cats );

		set_transient( 'business_one_page_pro_categories', $all_the_cool_cats );
	}

	if ( $all_the_cool_cats > 1 ) {
		// This blog has more than 1 category so business_one_page_pro_categorized_blog should return true.
		return true;
	} else {
		// This blog has only 1 category so business_one_page_pro_categorized_blog should return false.
		return false;
	}
}

if( ! function_exists( 'business_one_page_pro_excerpt' ) ):  
/**
 * business_one_page_pro_excerpt can truncate a string up to a number of characters while preserving whole words and HTML tags
 *
 * @param string $text String to truncate.
 * @param integer $length Length of returned string, including ellipsis.
 * @param string $ending Ending to be appended to the trimmed string.
 * @param boolean $exact If false, $text will not be cut mid-word
 * @param boolean $considerHtml If true, HTML tags would be handled correctly
 *
 * @return string Trimmed string.
 * 
 * @link http://alanwhipple.com/2011/05/25/php-truncate-string-preserving-html-tags-words/
 */
function business_one_page_pro_excerpt($text, $length = 100, $ending = '...', $exact = false, $considerHtml = true) {
	$text = strip_shortcodes( $text );
    $text = business_one_page_pro_strip_single( 'img', $text );
    $text = business_one_page_pro_strip_single( 'a', $text );
    
    if ($considerHtml) {
		// if the plain text is shorter than the maximum length, return the whole text
		if (strlen(preg_replace('/<.*?>/', '', $text)) <= $length) {
			return $text;
		}
		// splits all html-tags to scanable lines
		preg_match_all('/(<.+?>)?([^<>]*)/s', $text, $lines, PREG_SET_ORDER);
		$total_length = strlen($ending);
		$open_tags = array();
		$truncate = '';
		foreach ($lines as $line_matchings) {
			// if there is any html-tag in this line, handle it and add it (uncounted) to the output
			if (!empty($line_matchings[1])) {
				// if it's an "empty element" with or without xhtml-conform closing slash
				if (preg_match('/^<(\s*.+?\/\s*|\s*(img|br|input|hr|area|base|basefont|col|frame|isindex|link|meta|param)(\s.+?)?)>$/is', $line_matchings[1])) {
					// do nothing
				// if tag is a closing tag
				} else if (preg_match('/^<\s*\/([^\s]+?)\s*>$/s', $line_matchings[1], $tag_matchings)) {
					// delete tag from $open_tags list
					$pos = array_search($tag_matchings[1], $open_tags);
					if ($pos !== false) {
					unset($open_tags[$pos]);
					}
				// if tag is an opening tag
				} else if (preg_match('/^<\s*([^\s>!]+).*?>$/s', $line_matchings[1], $tag_matchings)) {
					// add tag to the beginning of $open_tags list
					array_unshift($open_tags, strtolower($tag_matchings[1]));
				}
				// add html-tag to $truncate'd text
				$truncate .= $line_matchings[1];
			}
			// calculate the length of the plain text part of the line; handle entities as one character
			$content_length = strlen(preg_replace('/&[0-9a-z]{2,8};|&#[0-9]{1,7};|[0-9a-f]{1,6};/i', ' ', $line_matchings[2]));
			if ($total_length+$content_length> $length) {
				// the number of characters which are left
				$left = $length - $total_length;
				$entities_length = 0;
				// search for html entities
				if (preg_match_all('/&[0-9a-z]{2,8};|&#[0-9]{1,7};|[0-9a-f]{1,6};/i', $line_matchings[2], $entities, PREG_OFFSET_CAPTURE)) {
					// calculate the real length of all entities in the legal range
					foreach ($entities[0] as $entity) {
						if ($entity[1]+1-$entities_length <= $left) {
							$left--;
							$entities_length += strlen($entity[0]);
						} else {
							// no more characters left
							break;
						}
					}
				}
				$truncate .= substr($line_matchings[2], 0, $left+$entities_length);
				// maximum lenght is reached, so get off the loop
				break;
			} else {
				$truncate .= $line_matchings[2];
				$total_length += $content_length;
			}
			// if the maximum length is reached, get off the loop
			if($total_length>= $length) {
				break;
			}
		}
	} else {
		if (strlen($text) <= $length) {
			return $text;
		} else {
			$truncate = substr($text, 0, $length - strlen($ending));
		}
	}
	// if the words shouldn't be cut in the middle...
	if (!$exact) {
		// ...search the last occurance of a space...
		$spacepos = strrpos($truncate, ' ');
		if (isset($spacepos)) {
			// ...and cut the text in this position
			$truncate = substr($truncate, 0, $spacepos);
		}
	}
	// add the defined ending to the text
	$truncate .= $ending;
	if($considerHtml) {
		// close all unclosed html-tags
		foreach ($open_tags as $tag) {
			$truncate .= '</' . $tag . '>';
		}
	}
	return $truncate;
}
endif; // End function_exists

/**
 * Callback function for Comment List *
 * 
 * @link https://codex.wordpress.org/Function_Reference/wp_list_comments 
 */
function business_one_page_pro_theme_comment( $comment, $args, $depth ){
    $GLOBALS['comment'] = $comment;
	extract( $args, EXTR_SKIP );

	if ( 'div' == $args['style'] ) {
		$tag = 'div';
		$add_below = 'comment';
	} else {
		$tag = 'li';
		$add_below = 'div-comment';
	}
?>
	<<?php echo $tag ?> <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ) ?> id="comment-<?php comment_ID() ?>">
	<?php if ( 'div' != $args['style'] ) : ?>
	<div id="div-comment-<?php comment_ID() ?>" class="comment-body">
	<?php endif; ?>
	<div class="comment-author vcard">
	<?php if ( $args['avatar_size'] != 0 ) echo get_avatar( $comment, $args['avatar_size'] ); ?>
	<?php printf( __( '<b class="fn">%s</b>', 'business-one-page-pro' ), get_comment_author_link() ); ?>
	</div>
	<?php if ( $comment->comment_approved == '0' ) : ?>
		<em class="comment-awaiting-moderation"><?php _e( 'Your comment is awaiting moderation.', 'business-one-page-pro' ); ?></em>
		<br />
	<?php endif; ?>

	<div class="comment-metadata commentmetadata">
    <a href="<?php echo htmlspecialchars( get_comment_link( $comment->comment_ID ) ); ?>">
		<time>
        <?php
			/* translators: 1: date, 2: time */
			printf( __( '%1$s - %2$s', 'business-one-page-pro' ), get_comment_date( 'M n, Y' ), get_comment_time() ); ?>
        </time>
    </a>
	</div>
    
    <div class="comment-content"><?php comment_text(); ?></div>
    
	<div class="reply">
	<?php comment_reply_link( array_merge( $args, array( 'add_below' => $add_below, 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
	</div>
	<?php if ( 'div' != $args['style'] ) : ?>
	</div>
	<?php endif; ?>
<?php
}

/**
 * Function to get Sections 
 * 
 * @return Enables Sections
 */
function business_one_page_pro_get_sections(){
    $enabled_section = array();
    
    if( get_theme_mod( 'business_one_page_pro_sort_homepage' ) ){
        $sections = get_theme_mod( 'business_one_page_pro_sort_homepage' );
    }else{
        $sections = array( 'about', 'services', 'cta1', 'portfolio', 'team', 'clients', 'blog', 'testimonial', 'cta2', 'contact' );
    }
    
    foreach ( $sections as $section ){
        $enabled_section[] = array(
            'id' => $section,
            'menu_text' => esc_attr( get_theme_mod( 'business_one_page_pro_' . $section . '_section_menu_title' ) ),
        );
    }
    return $enabled_section;  
}

/**
 * Function to list dynamic sidebar
*/
function business_one_page_pro_get_dynamnic_sidebar( $nosidebar = false, $sidebar = false, $default = false ){
    $sidebar_arr = array();
    $sidebars = get_theme_mod( 'business_one_page_pro_sidebar' );
    
    if( $default ) $sidebar_arr['default-sidebar'] = __( 'Default Sidebar', 'business-one-page-pro' );
    if( $sidebar ) $sidebar_arr['sidebar'] = __( 'Sidebar', 'business-one-page-pro' );
    
    if( $sidebars ){
        foreach( $sidebars as $sidebar ){
            $sidebar_arr[$sidebar['id']] = $sidebar['name'];
        }
    }
    
    if( $nosidebar ) $sidebar_arr['no-sidebar'] = __( 'No Sidebar', 'business-one-page-pro' );
    
    return $sidebar_arr;
}

/**
 * Function to list Custom Pattern
*/
function business_one_page_pro_get_patterns(){
    $patterns = array();
    $patterns['nobg'] = get_template_directory_uri() . '/images/patterns_thumb/' . 'nobg.png';
    for( $i=0; $i<38; $i++ ){
        $patterns['pattern'.$i] = get_template_directory_uri() . '/images/patterns_thumb/' . 'pattern' . $i .'.png';
    }
    for( $j=1; $j<26; $j++ ){
        $patterns['hbg'.$j] = get_template_directory_uri() . '/images/patterns_thumb/' . 'hbg' . $j . '.png';
    }
    return $patterns;
}

/**
 * Function to populate list of social Icons
*/
function business_one_page_pro_social_icons(){
    $social_icons = array();
    
    $social_icons['dribbble']   = __( 'Dribbble', 'business-one-page-pro' );
    $social_icons['facebook']   = __( 'Facebook', 'business-one-page-pro' );
    $social_icons['foursquare'] = __( 'Foursquare', 'business-one-page-pro' );
    $social_icons['flickr']     = __( 'Flickr', 'business-one-page-pro' );
    $social_icons['google-plus']= __( 'Google Plus', 'business-one-page-pro' );
    $social_icons['instagram']  = __( 'Instagram', 'business-one-page-pro' );
    $social_icons['linkedin']   = __( 'LinkedIn', 'business-one-page-pro' );
    $social_icons['pinterest']  = __( 'Pinterest', 'business-one-page-pro' );
    $social_icons['reddit']     = __( 'Reddit', 'business-one-page-pro' );
    $social_icons['skype']      = __( 'Skype', 'business-one-page-pro' );
    $social_icons['stumbleupon']= __( 'StumbleUpon', 'business-one-page-pro' );
    $social_icons['tumblr']     = __( 'Tumblr', 'business-one-page-pro' );
    $social_icons['twitter']    = __( 'Twitter', 'business-one-page-pro' );
    $social_icons['vimeo']      = __( 'Vimeo', 'business-one-page-pro' );
    $social_icons['youtube']    = __( 'YouTube', 'business-one-page-pro' );
    
    return $social_icons;
}

/**
 * Retrieves the Attachment ID from the file URL
 *  
 * @link https://pippinsplugins.com/retrieve-attachment-id-from-image-url/
 */
function business_one_page_pro_get_image_id( $image_url ){
	global $wpdb;
	$attachment = $wpdb->get_col( $wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE guid='%s';", $image_url ) ); 
    if( $attachment ) return $attachment[0]; 
}

/**
 * List out font awesome icon list
*/
function business_one_page_pro_get_icon_list(){
    include get_template_directory() . '/inc/fontawesome.php';
    echo '<div class="rara-font-awesome-list"><ul class="rara-font-group">';
    foreach( $fontawesome as $font ){
        echo '<li><i class="fa ' . esc_attr( $font ) . '"></i></li>';
    }
    echo '</ul></div>';
    
}

/**
 * Helper Function for Image widget
*/
function business_one_page_pro_get_image_field( $id, $name, $image, $label ){
    
    $output = '';
    
    $output .= '<div class="widget-upload">';
    $output .= '<label for="' . $id . '">' . esc_html( $label ) . '</label><br/>';
    $output .= '<input id="' . $id . '" class="rara-upload" type="text" name="' . $name . '" value="' . $image . '" placeholder="' . __('No file chosen', 'business-one-page-pro') . '" />' . "\n";
    if ( function_exists( 'wp_enqueue_media' ) ) {
        if ( $image == '' ) {
            $output .= '<input id="upload-' . $id . '" class="rara-upload-button button" type="button" value="' . __('Upload', 'business-one-page-pro') . '" />' . "\n";
        } else {
            $output .= '<input id="upload-' . $id . '" class="rara-upload-button button" type="button" value="' . __('Change', 'business-one-page-pro') . '" />' . "\n";
        }
    } else {
        $output .= '<p><i>' . __('Upgrade your version of WordPress for full media support.', 'business-one-page-pro') . '</i></p>';
    }

    $output .= '<div class="rara-screenshot" id="' . $id . '-image">' . "\n";

    if ( $image != '' ) {
        $remove = '<a class="rara-remove-image"></a>';
        $attachment_id = business_one_page_pro_get_image_id( $image );
        $image_array = wp_get_attachment_image_src( $attachment_id, 'full');
        $image = preg_match('/(^.*\.jpg|jpeg|png|gif|ico*)/i', $image);
        if ( $image ) {
            $output .= '<img src="' . esc_url( $image_array[0] ) . '" alt="" />' . $remove;
        } else {
            // Standard generic output if it's not an image.
            $output .= '<small>' . __( 'Please upload valid image file.', 'business-one-page-pro' ) . '</small>';
        }     
    }
    $output .= '</div></div>' . "\n";
    
    echo $output;
}

/**
 * Client Logo Slider
*/
function business_one_page_pro_client_logo( $home = true ){
    $title = $home ? get_theme_mod( 'business_one_page_pro_clients_section_title' ) : get_theme_mod( 'business_one_page_pro_our_client_title' );
    $bg    = $home ? get_theme_mod( 'business_one_page_pro_clients_section_bg' ) : get_theme_mod( 'business_one_page_pro_our_clients_bg' );
    $client_qry = new WP_Query( array( 'post_type' => 'logo', 'posts_per_page' => -1, 'post_status' => 'publish' ) );
    
    if( $client_qry->have_posts() || $title ){  
    ?>
    <section id="clients" class="our-clients" <?php if( $bg ) echo 'style="background: url(' . esc_url( $bg ) .') no-repeat; background-size: cover;"' ?> >
    	<div class="container">
    		<?php 
                
                if( $title ) echo '<h2 class="title">' . esc_html( $title ) . '</h2>'; 
                
                if( $client_qry->have_posts() ){        
                ?>
                    <ul id="client-slider">
                    <?php
                        while( $client_qry->have_posts() ){
                            $client_qry->the_post();
                            
                            $link = get_post_meta( get_the_ID(), '_business_one_page_pro_logo_link', true );
                            echo '<li>';
                            if( $link ) echo '<a href="' . esc_url( $link ) . '" target="_blank" title="' . esc_attr( get_the_title() ) . '">';
                            
                            the_post_thumbnail( 'full' );
                            
                            if( $link ) echo '</a>';
                            
                            echo '</li>';        
                        }
                    ?>
                    </ul>    
                <?php
                }
                ?>
    	</div>
    </section>
    <?php
    }    
}

/**
 *  Custom Pagination
*/
function business_one_page_pro_pagination(){
    $pagination = get_theme_mod( 'business_one_page_pro_pagination_type', 'default' );
    
    switch( $pagination ){
        case 'default': // Default Pagination
        
        the_posts_navigation();
        
        break;
        
        case 'numbered': // Numbered Pagination
        
        the_posts_pagination( array(
            'prev_text'          => __( 'Previous', 'business-one-page-pro' ),
            'next_text'          => __( 'Next', 'business-one-page-pro' ),
            'before_page_number' => '<span class="meta-nav screen-reader-text">' . __( 'Page', 'business-one-page-pro' ) . ' </span>',
         ) );
        
        break;
        
        case 'load_more': // Load More Button
        case 'infinite_scroll': // Auto Infinite Scroll
        
        echo '<div class="pagination"></div>';
        
        break;
        
        default:
        
        the_posts_navigation();
        
        break;
    }   
}

/**
 * Function to retrive page specific sidebar and corresponding body class
 * 
 * @param boolean $sidebar
 * @param boolean $class
 * 
 * @return string dynamic sidebar id / classname
*/
function business_one_page_pro_sidebar( $sidebar = false, $class = false ){
    
    global $post;
    $return = false;
    
    if( ( is_front_page() && is_home() ) || is_home() ){
        //blog/home page 
        $blog_sidebar = get_theme_mod( 'business_one_page_pro_blog_page_sidebar', 'sidebar' );
        $blog_layout  = get_theme_mod( 'business_one_page_pro_blog_layout', 'default' );
        $layout       = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( is_active_sidebar( $blog_sidebar ) ){            
            if( $sidebar ) $return = $blog_sidebar; //With Sidebar
            if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; 
            if( $class && $layout == 'left-sidebar' )  $return = 'leftsidebar';    
            
            if( $blog_layout == 'full' ){
                $return .= ' blog-large';
            }elseif( $blog_layout == 'square' ){
                $return .= ' blog-medium';
            }elseif( $blog_layout == 'circular'  ){
                $return .= ' blog-round';
            }        
        }else{
            if( $sidebar ) $return = false; //Fullwidth
            if( $class ) $return = 'full-width';
            
            if( $blog_layout == 'full' ){
                $return .= ' blog-large';
            }elseif( $blog_layout == 'square' ){
                $return .= ' blog-medium';
            }elseif( $blog_layout == 'circular'  ){
                $return .= ' blog-round';
            }
        }        
    }
    
    if( is_archive() ){
        //archive page
        $archive_sidebar = get_theme_mod( 'business_one_page_pro_archive_page_sidebar', 'sidebar' );
        $cat_sidebar     = get_theme_mod( 'business_one_page_pro_cat_archive_page_sidebar', 'sidebar' );
        $tag_sidebar     = get_theme_mod( 'business_one_page_pro_tag_archive_page_sidebar', 'sidebar' );
        $date_sidebar    = get_theme_mod( 'business_one_page_pro_date_archive_page_sidebar', 'sidebar' );
        $author_sidebar  = get_theme_mod( 'business_one_page_pro_author_archive_page_sidebar', 'sidebar' );
        $layout          = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( is_category() ){
            
            if( $cat_sidebar == 'no-sidebar' || ( $cat_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( $cat_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $cat_sidebar ) ){
                if( $sidebar ) $return = $cat_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
                
        }elseif( is_tag() ){
            
            if( $tag_sidebar == 'no-sidebar' || ( $tag_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( ( $tag_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $tag_sidebar ) ){
                if( $sidebar ) $return = $tag_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';              
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
            
        }elseif( is_author() ){
            
            if( $author_sidebar == 'no-sidebar' || ( $author_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( ( $author_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $author_sidebar ) ){
                if( $sidebar ) $return = $author_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
            
        }elseif( is_date() ){
            
            if( $date_sidebar == 'no-sidebar' || ( $date_sidebar == 'default-sidebar' && $archive_sidebar == 'no-sidebar' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( ( $date_sidebar == 'default-sidebar' && $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $date_sidebar ) ){
                if( $sidebar ) $return = $date_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }                         
            
        }else{
            if( $archive_sidebar != 'no-sidebar' && is_active_sidebar( $archive_sidebar ) ){
                if( $sidebar ) $return = $archive_sidebar;
                if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
            }                      
        }
        
    }
    
    if( is_singular() ){
        $post_sidebar = get_theme_mod( 'business_one_page_pro_single_post_sidebar', 'sidebar' );
        $page_sidebar = get_theme_mod( 'business_one_page_pro_single_page_sidebar', 'sidebar' );
        $layout       = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( get_post_meta( $post->ID, '_business_one_page_pro_sidebar', true ) ){
            $single_sidebar = get_post_meta( $post->ID, '_business_one_page_pro_sidebar', true );
        }else{
            $single_sidebar = 'default-sidebar';
        }

        if( get_post_meta( $post->ID, '_business_one_page_pro_sidebar_layout', true ) ){
            $sidebar_layout = get_post_meta( $post->ID, '_business_one_page_pro_sidebar_layout', true );
        }else{
            $sidebar_layout = 'default-sidebar';
        }
        
        if( is_page() ){
            
            if( is_page_template( 'templates/template-home.php' ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }else{
                if( ( $single_sidebar == 'no-sidebar' ) || ( ( $single_sidebar == 'default-sidebar' ) && ( $page_sidebar == 'no-sidebar' ) ) ){
                    if( $sidebar ) $return = false; //Fullwidth
                    if( $class ) $return = 'full-width';
                }elseif( $single_sidebar == 'default-sidebar' && $page_sidebar != 'no-sidebar' && is_active_sidebar( $page_sidebar ) ){
                    if( $sidebar ) $return = $page_sidebar;
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
                }elseif( is_active_sidebar( $single_sidebar ) ){
                    if( $sidebar ) $return = $single_sidebar;
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                    if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
                }else{
                    if( $sidebar ) $return = false; //Fullwidth
                    if( $class ) $return = 'full-width';
                }
            }
        }
        
        if( is_single() ){
            
            if( ( $single_sidebar == 'no-sidebar' ) || ( ( $single_sidebar == 'default-sidebar' ) && ( $post_sidebar == 'no-sidebar' ) ) ){
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }elseif( $single_sidebar == 'default-sidebar' && $post_sidebar != 'no-sidebar' && is_active_sidebar( $post_sidebar ) ){
                if( $sidebar ) $return = $post_sidebar;
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
            }elseif( is_active_sidebar( $single_sidebar ) ){
                if( $sidebar ) $return = $single_sidebar;
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) ) $return = 'rightsidebar';
                if( $class && ( ( $sidebar_layout == 'default-sidebar' && $layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) ) $return = 'leftsidebar';
            }else{
                if( $sidebar ) $return = false; //Fullwidth
                if( $class ) $return = 'full-width';
            }
        }
    }
    
    if( is_search() ){
        $search_sidebar = get_theme_mod( 'business_one_page_pro_search_page_sidebar', 'sidebar' );
        $layout         = get_theme_mod( 'business_one_page_pro_layout_style', 'right-sidebar' );
        
        if( $search_sidebar != 'no-sidebar' && is_active_sidebar( $search_sidebar ) ){
            if( $sidebar ) $return = $search_sidebar;
            if( $class && $layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
            if( $class && $layout == 'left-sidebar' ) $return = 'leftsidebar';
        }else{
            if( $sidebar ) $return = false; //Fullwidth
            if( $class ) $return = 'full-width';
        }
        
    }
    
    return $return;        
}

/**
 * Function to get the post view count 
 */
function business_one_page_pro_get_views( $post_id ){
    $count_key = '_business_one_page_pro_view_count';
    $count = get_post_meta( $post_id, $count_key, true );
    if( $count == '' ){        
        return __( "0 View", 'business-one-page-pro' );
    }elseif($count<=1){
        return $count. __(' View', 'business-one-page-pro' );
    }else{
        return $count. __(' Views', 'business-one-page-pro' );    
    }    
}

/**
 * Function to add the post view count 
 */
function business_one_page_pro_set_views( $post_id ) {
    $count_key = '_business_one_page_pro_view_count';
    $count = get_post_meta( $post_id, $count_key, true );
    if( $count == '' ){
        $count = 0;
        delete_post_meta( $post_id, $count_key );
        add_post_meta( $post_id, $count_key, '1' );
    }else{
        $count++;
        update_post_meta( $post_id, $count_key, $count );
    }
}

/**
 * 
 * @link http://www.altafweb.com/2011/12/remove-specific-tag-from-php-string.html
*/
function business_one_page_pro_strip_single( $tag, $string ){
    $string=preg_replace('/<'.$tag.'[^>]*>/i', '', $string);
    $string=preg_replace('/<\/'.$tag.'>/i', '', $string);
    return $string;
}

/**
 * convert hex to rgb
 * @link http://bavotasan.com/2011/convert-hex-color-to-rgb-using-php/
*/
function business_one_page_pro_hex2rgb($hex) {
   $hex = str_replace("#", "", $hex);

   if(strlen($hex) == 3) {
      $r = hexdec(substr($hex,0,1).substr($hex,0,1));
      $g = hexdec(substr($hex,1,1).substr($hex,1,1));
      $b = hexdec(substr($hex,2,1).substr($hex,2,1));
   } else {
      $r = hexdec(substr($hex,0,2));
      $g = hexdec(substr($hex,2,2));
      $b = hexdec(substr($hex,4,2));
   }
   $rgb = array($r, $g, $b);
   //return implode(",", $rgb); // returns the rgb values separated by commas
   return $rgb; // returns an array with the rgb values
}

/**
 * Query WooCommerce activation
 */
function is_woocommerce_activated() {
	return class_exists( 'woocommerce' ) ? true : false;
}

/**
 * Query Contact Form 7
 */
function is_cf7_activated() {
	return class_exists( 'WPCF7' ) ? true : false;
}

/**
 * Query Jetpack activation
*/
function is_jetpack_activated( $gallery = false ){
	if( $gallery ){
        return ( class_exists( 'jetpack' ) && Jetpack::is_module_active( 'tiled-gallery' ) ) ? true : false;
	}else{
        return class_exists( 'jetpack' ) ? true : false;
    }           
}